"use strict";
/***********************************************************/
/*  BitSight Archer Integration - Subscriptions            */
/***********************************************************/

/*
Purpose: 
	Updates Archer Portfolio records based on the current BitSight Portfolio settings for active TRM or RM subscriptions including headline ratings and vectors if using TRM.
	NOTE: This is also a somewhat self-healing application resetting any portfolios out-of-sync with the BitSight Portal AND can be used for initial setup for existing BitSight Portal customers.

High Level Overview:
	Obtains list of Archer BitSight Portfolio records (used to match BitSight data)
	Obtains current BitSight Company subscriptions for TRM/RM
	Iterates through companies to retrieve applicable rating information from BitSight API
	Consolodates appropriate data based on subscription
	Uploads results to Archer

Detailed Process: 
	1. Logs into Archer to get session token
	2. Obtain list of existing Archer Portfolio records (must know due to POST vs PUT)
	3. Retrive Customer Name from BitSight to pass into x-BitSight-Customer header param
	4. Logs into BitSight API to get companies
	5. Consolidate Archer records with BitSight Companies for any newly licensed companies to track from BitSight portal
	6. Iterates through the companies to get guid/subscriptions
	7. Iterates requests to BitSight API to obtain rating information for each company
	8. Add domain, description, and industry from BitSight info
	9. Obtains the FieldIDs for the content post via api from the app
	10. Obtain values list IDs programmatically	
	11. Handle POST for new vs PUT for updating existing
	12. Handling the new subscription type and lack of detailed ratings.
	13. Need to ensure we don't overwrite the ratings to N/A in case the user had ratings...
	14. Handling different subscription types in PUT for existing records 
	15. Generates the postbody array for each company found
	16. Iterate and create/update BitSight Portfolio Records via API to Archer
*/

/**********************************************************/
/* VERSIONING                                             */
/**********************************************************/
/*  

	2/22/2025 - Version 2.3
	Added the Tier field - Note that an additional Tier values list field needs to be added to Archer. 

	8/6/2024 - Version 2.2
	Added error handling since alert-only or companies may be empty - when originally developed the JSON block existed whether data existed or it was empty.
	
	5/20/2024 - Version 2.1
	Added "Company ID" to be synced

	1/12/2022 - Version 2.0
    Initial Version - 
*/

/********************************************************/
/********	ARCHER DFM APPROVED LIBRARIES
/********************************************************/
var request = require("request");
//var xpath = require('xpath');
var xmldom = require("xmldom");
var xml2js = require("xml2js");

/********************************************************/
/***** DEFAULT PARAMS
/********************************************************/
/* These params will be used unless overriden by parameters passed in. */
var defaultParams = {
	"archer_username": "dfm_BitSightAPINightly", //Example used in Installation Guide. Update name if you change it.
	"archer_password": "YOURPASSWORD",
	"archer_instanceName": "INSTANCENAME", //Example: 88555 or Dev						Instance name is case-sensitive. SaaS instances are typically numbers
	"archer_webroot": "https://ARCHERURL.com/", //Example: https://[ArcherDomainURL]/		Must include trailing slash
	"archer_webpath": "RSAarcher/", //Example: RSAarcher/						Must include trailing slash UNLESS there is no virtual directory...then leave blank - Preconfigured for Archer SaaS environment
	"archer_ws_root": "RSAarcher/ws/", //Example: RSAarcher/ws/					Must include trailing slash and vdir is case-sensitive - Preconfigured for Archer SaaS environment
	"archer_rest_root": "RSAArcher/platformapi/", //Example: RSAarcher/platformapi/			Must include trailing slash and vdir is case-sensitive - Preconfigured for Archer SaaS environment

	"bitsight_token": "YOUR_BITSIGHT_TOKEN", //BitSight API Token for authentication

	//Field GUID of the Bitsight Porfolio [Tier] field. This new field was added on 2/22/2025.
	//In the event a customer configured their BitSight Portfolio app, upgraded from V2.0-2.2, or didn't want to package, this GUID can be updated here.
	"archer_bitsightportfolio_tier_guid": "be5d0437-368f-4f78-9d6c-902c439a3288",

	//Web call options
	"proxy": "",
	"verifyCerts": "true",

	//Application Names
	"archer_ThirdPartyProfileApp": "Third Party Profile", //Name of the "Third Party Profile" application in Archer for your organization.
	"archer_BitSightPortfolioApp": "BitSight Portfolio", //Name of the "BitSight Portfolio" ODA in Archer for your organization.

	//Typically these will not change
	"bitsight_webroot": "https://api.bitsighttech.com/", //Example: https://api.bitsighttech.com/		Main BitSight API URL				*Must include trailing slash
	"bitsight_subscriptions": "ratings/v1/subscriptions", //Example: ratings/v1/subscriptions				Endpoint for current subscriptions (3 types)
	"bitsight_currentuser": "users/current", //Example: users/current 						Endpoint for getting current user/company info
	"bitsight_companyratings": "ratings/v1/companies", //Example: ratings/v1/companies					Endpoint for ratings for a specific company
	"bitsight_weblink": "https://service.bitsighttech.com/app/spm/company/", //Example: https://service.bitsighttech.com/app/spm/company/			URL before the Company GUID
	"bitsight_tiers": "ratings/v1/tiers", //Example: ratings/v1/tiers				Endpoint for obtaining the tiers and the companies in those tiers - Added 2/22/2025

	//Typically these will not change unless Archer changes the structure
	"archer_loginpath": "general.asmx", //Example: general.asmx
	"archer_searchpath": "search.asmx", //Example: search.asmx
	"archer_contentpath": "core/content", //Example: core/content				//Do NOT include trailing slash
	"archer_applicationpath": "core/system/application/", //Example: core/system/application/							Must include trailing slash
	"archer_fielddefinitionapppath": "core/system/fielddefinition/application/", //Example: core/system/fielddefinition/application/			Must include trailing slash
	"archer_fielddefinitionpath": "core/system/fielddefinition/", //Example: core/system/fielddefinition/						Must include trailing slash
	"archer_valueslistvaluepath": "core/system/valueslistvalue/flat/valueslist/", //Example: core/system/valueslistvalue/flat/valueslist/		Must include trailing slash
	"archer_version": "core/system/applicationinfo/version", //Example: core/system/applicationinfo/version				Do NOT include trailing slash

	//*****************DO NOT MODIFY************************
	"bIsSaaS": "true", //**Must be set to true! In the future, this may be converted to a JST data feed
	"bDFDebugging": "false", //**Must be set to false !Only applicable for Data Feed Jobs. Always set to false for SaaS.
};

var bVerboseLogging = true; //Verbose logging will log the post body data and create output files which takes up more disk space. Very useful for troubleshooting, but not necessary unless you are having issues.
var bIsLegacyIntegration = false; //Set this to true if you are using the Archer 6.7 version of the package. If you are using Archer 6.10 P1 or higher, set this to: false

var SaaSParams = {}; //This is automatically populated with all the Archer field IDs and values list IDs making deployment between environments easy.
var appPrefix = "200"; //Used to store files in logs subdirectory when VerboseLogging is enabled.

//bSkipCompanyIfError
//If set to true, will skip over a BitSight company rating request for URL5 (logging a warning) instead of erroring/stopping the entire process.
//We recommend setting this to true to ensure the vast majority of requests work. Occassionally a single API request could timeout and probably shouldn't terminate the entire process.
//var bSkipCompanyIfError = true;
var bSkipCompanyIfError = false;

var httpTimeout = 60000; //Global setting for the http timeout per http request. Default: 60000

/********************************************************/
/********	DEBUGGING/TESTING SETTINGS
/********************************************************/
var testingMode_Tokens = {
	"LastRunTime": "2018-06-13T18:31:41Z",
	"PreviousRunContext": "",
};

var testingMode_Params = {};

/********************************************************/
/***** GLOBAL VARS
/********************************************************/
var testingMode = true;
var sSessionToken = null;
var params = null;
var tokens = null;
var verifyCerts = false;
var useProxy = false;
var errorArray = [];
var previousRunContext = {};
var aBitSightSubscriptions = [];
var aBitSightRatings = []; //Final object returned to Archer data feed
var aBitSightTierInfo = [];
var ArcherValuesLists = []; //This will be used for iterating through to obtain values list value id from Archer
var bSaaSErrorFound = false;
var sDebug = "";
var iCurrentCompany = 0;
var aFinalPostBody = [];
var sArcherVersionNum = "TBD";

//Just a simple way to remember the Archer input data types
var ArcherInputTypes = {
	"text": 1,
	"numeric": 2,
	"date": 3,
	"valueslist": 4,
	"externallinks": 7,
	"usergrouprecperm": 8,
	"crossref": 9,
	"attachment": 11,
	"image": 12,
	"matrix": 16,
	"ipaddress": 19,
	"relatedrecords": 23,
	"subform": 24,
};

var COST_ArcherBitSightVersion = "RSA Archer 1.0";
var COST_Platform = "RSA Archer";
var BitSightCustomerName = "unknown";

/********************************************************/
/********	INIT
/********************************************************/
function init() {
	/* run the feed */
	//LogInfo("Datafeed Init");
	/* check if testing mode should be active (no archer DFE present) */
	if (typeof context == "undefined") {
		testingMode = true;
	}
	/* get params and tokens */
	params = getArcherParams();
	tokens = getArcherTokens();
	/* setup cert verify */
	if (params["verifyCerts"].toLowerCase() == "true") {
		verifyCerts = true;
	} else {
		process.env.NODE_TLS_REJECT_UNAUTHORIZED = "0";
	}
	/* setup proxy */
	if (params["proxy"] != null && params["proxy"] != "") {
		useProxy = true;
	}
	/* get the last run time */
	var lrt = new Date("1970-01-10T00:00:00Z");
	/* check for last run time */
	if ("LastRunTime" in tokens && tokens["LastRunTime"] != null && tokens["LastRunTime"] != "") {
		lrt = new Date(tokens["LastRunTime"]);
	}
	//LogInfo("Last Datafeed Run Time: " + lrt.toISOString());
	return lrt;
}

/********************************************************/
/********	DATA CONVERSIONS
/********************************************************/
function jsonToXMLString(json, rootElement) {
	if (rootElement) {
		var bldrOpts = {
			headless: true,
			rootName: rootElement,
			renderOpts: { "pretty": true, "indent": "    ", "newline": "\r\n", "cdata": true },
		};
	} else {
		var bldrOpts = {
			headless: true,
			renderOpts: { "pretty": true, "indent": "    ", "newline": "\r\n", "cdata": true },
		};
	}
	return new xml2js.Builder(bldrOpts).buildObject(json);
}

function jsonToString(json) {
	return JSON.stringify(json, null, 4);
}

function xmlStringToJSON(xml, callBack) {
	xml2js.parseString(xml, {}, callBack);
}

function xmlStringToXmlDoc(xml) {
	var p = new xmldom.DOMParser();
	return p.parseFromString(xml);
}

function jsonArrayToXMLBuffer(jsonArray, elementName) {
	/* holds the buffers */
	var buffers = [];
	/* conver each element to an xml buffer */
	for (var i in jsonArray) {
		/* convert it to xml */
		var xmlString = jsonToXMLString(jsonArray[i], elementName);
		/* convert it to a buffer */
		var b = Buffer.from(xmlString + "\n", "utf8");
		/* add to buffers array */
		buffers.push(b);
	}
	/* concat to one giant buffer */
	return Buffer.concat(buffers);
}

/********************************************************/
/********	ARCHER CALLBACK INTERFACE
/********************************************************/
function getArcherParams() {
	var params = null;
	/* which params should we use? */
	if (testingMode) {
		params = testingMode_Params;
	} else {
		params = context.CustomParameters;
	}
	/* now add the default params */
	for (var k in defaultParams) {
		if (!(k in params)) {
			params[k] = defaultParams[k];
		}
	}
	/* return them */
	return params;
}

function getArcherTokens() {
	if (testingMode) {
		return testingMode_Tokens;
	}
	return context.Tokens;
}

var dfCallback = function (err, data) {
	if (err) {
		LogError("ERROR: Datafeed Failure due to error.");
		if (params["bIsSaaS"] == "true") {
			LogSaaSError();
		} else {
			if (params["bDFDebugging"] == "true" || params["bDFDebugging"] == true) {
				LogInfo("Data Feed Debugging Enabled");
				//The commented line sends the data to a record in Archer, but the record ID needs to be provided.
				var start = Buffer.from("<Records><Record><![CDATA[\n", "utf8");
				var middle = Buffer.from(sDebug, "utf8");
				//var end = Buffer.from(']]></Report_Results_HTML></Record></Records>\n', 'utf8');
				var end = Buffer.from("]]></Record></Records>\n", "utf8");
				var returntoArcher = Buffer.concat([start, middle, end]);
				var sXMLReturn = returntoArcher.toString("utf8");
				sendDataToArcher(sXMLReturn);
			} else {
				sendDataToArcher(null);
			}
		}
	} else {
		if (params["bIsSaaS"] == "true") {
			LogSaaSSuccess();
			LogInfo("SUCCESS: Overall Success");
		} else {
			LogInfo("Exporting Data to Archer.");
			sendDataToArcher(data);
		}
	}
};

function sendDataToArcher(data) {
	if (testingMode || bVerboseLogging) {
		if (data) {
			var fs = require("fs");
			fs.writeFile("logs\\" + appPrefix + "\\" + "!ReturnedToArcher.xml", data, function (err) {
				if (err) {
					LogError("ERROR SAVING FILE IN TEST MODE: " + err);
				}
			});
		}
		if (errorArray.length > 0) {
			for (var i in errorArray) {
				LogError(errorArray[i]);
			}
		}
	} else {
		/* check for any errors */
		if (errorArray.length > 0) {
			//callback(errorArray,{output: data, previousRunContext: JSON.stringify(previousRunContext)}); //Only if data came back as JSON
			callback(errorArray, { output: data, previousRunContext: JSON.stringify(previousRunContext) }); //Attempting to use the XML returned from the Archer API
		} else {
			//callback(null,{output: data, previousRunContext: JSON.stringify(previousRunContext)});  //Only if data came back as JSON
			callback(null, { output: data, previousRunContext: JSON.stringify(previousRunContext) }); //Attempting to use the XML returned from the Archer API
		}
	}
}

/********************************************************/
/********	ERROR HANDLING AND LOGGING
/********************************************************/
function captureError(text) {
	LogSaaSError();

	// if(text!=null)
	// {
	// 	/* create a new error to get the stack */
	// 	var e = new Error();
	// 	/* create error string for array */
	// 	var errString = text + "\n" + e.stack;
	// 	/* add to error array */
	// 	errorArray.push(errString);
	// }
}

function getDateTime() {
	var dt = new Date();
	return (
		pad(dt.getFullYear(), 4) +
		"-" +
		pad(dt.getMonth() + 1, 2) +
		"-" +
		pad(dt.getDate(), 2) +
		" " +
		pad(dt.getHours(), 2) +
		":" +
		pad(dt.getMinutes(), 2) +
		":" +
		pad(dt.getSeconds(), 2)
	);
}

function LogInfo(text) {
	console.log(getDateTime() + " :: INFO  :: " + text);
	if (params["bDFDebugging"] == "true" || params["bDFDebugging"] == true) {
		appendDebug(text);
	}
}

function LogError(text) {
	LogSaaSError();
	console.log(getDateTime() + " :: ERROR :: " + text);
	if (params["bDFDebugging"] == "true" || params["bDFDebugging"] == true) {
		appendDebug(text);
	}
}

function LogWarn(text) {
	console.log(getDateTime() + " :: WARN  :: " + text);
	if (params["bDFDebugging"] == "true" || params["bDFDebugging"] == true) {
		appendDebug(text);
	}
}

function LogVerbose(text) {
	if (bVerboseLogging) {
		LogInfo(text);
	}
}

function pad(num, size) {
	var s = num + "";
	while (s.length < size) s = "0" + s;
	return s;
}

/********************************************************/
/********	LogSaaSError
/********************************************************/
var LogSaaSError = function () {
	//Simple method to log an error to a file for SaaS for batch execution. The batch file will check if this file exists.
	if (params["bIsSaaS"] == "true" && bSaaSErrorFound == false) {
		bSaaSErrorFound = true; //Only log if we haven't found an error to help avoid file lock issues.
		var fs = require("fs");
		fs.writeFileSync("logs\\error" + "-" + appPrefix + ".txt", "ERROR");
		LogInfo("Logged error and created logs\\error.txt file.");
	}
};

/************************************************************************************************************************************************************************************************************************************/
/********	LogSaaSSuccess
/************************************************************************************************************************************************************************************************************************************/
var LogSaaSSuccess = function () {
	//Simple method to log successful execution for SaaS for batch execution. The batch file will check if this file exists.
	if (params["bIsSaaS"] == "true" && bSaaSErrorFound == false) {
		var fs = require("fs");
		fs.writeFileSync("logs\\success" + "-" + appPrefix + ".txt", "SUCCESS");
		LogInfo("Logged success and created logs\\success" + "-" + appPrefix + ".txt file.");
	}
};

//GetArcherText validates then returns data from a text value
function GetArcherText(sText) {
	if (typeof sText == "undefined" || typeof sText._ == "undefined" || sText == null) {
		return "";
	} else {
		return sText._;
	}
}

//GetArcherValue validates then returns data from a single value list
function GetArcherValue(jValueNode) {
	if (
		typeof jValueNode == "undefined" ||
		jValueNode == null ||
		typeof jValueNode.ListValues == "undefined" ||
		typeof jValueNode.ListValues[0] == "undefined" ||
		typeof jValueNode.ListValues[0].ListValue == "undefined" ||
		typeof jValueNode.ListValues[0].ListValue[0] == "undefined" ||
		typeof jValueNode.ListValues[0].ListValue[0]._ == "undefined"
	) {
		return "";
	} else {
		return jValueNode.ListValues[0].ListValue[0]._;
	}
}

/********************************************************/
/********	WEB CALL
/********************************************************/
function webCall(url, method, headers, postBody, callBack) {
	/* build options */
	var options = {
		method: method,
		uri: url,
		headers: headers,
		body: postBody,
		timeout: httpTimeout,
		rejectUnauthorized: verifyCerts,
	};
	/* add in proxy */
	if (useProxy) {
		options["proxy"] = params["proxy"];
	}

	//LogInfo("HERE url: " + url);
	//LogInfo("HERE method: " + method);
	//LogInfo("HERE postBody: " + postBody);

	/* make the request */
	request(options, function handleResponse(err, response, body) {
		var errMessage = null;
		var headers = {};
		/* check for error */
		if (err) {
			errMessage = "HTTP ERROR: " + err;
		} else {
			headers = response.headers;
			//LogInfo("HERE headers: " + headers);
			if (response.statusCode != 200) {
				errMessage = "HTTP " + response.statusCode + " ERROR: " + err;
			}
		}
		/* check for error */
		if (errMessage) {
			captureError("Error in HTTP Response: " + errMessage);
			callBack(true, headers, body);
		} else {
			callBack(false, headers, body);
		}
	});
}

function appendDebug(sTxt) {
	sDebug += sTxt.toString("utf8") + "\n";
	//sDebug+=sTxt + "|";
}

function b64Encode(str) {
	var CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	var out = "",
		i = 0,
		len = str.length,
		c1,
		c2,
		c3;

	while (i < len) {
		c1 = str.charCodeAt(i++) & 0xff;
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt((c1 & 0x3) << 4);
			out += "==";
			break;
		}

		c2 = str.charCodeAt(i++);
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
			out += CHARS.charAt((c2 & 0xf) << 2);
			out += "=";
			break;
		}

		c3 = str.charCodeAt(i++);
		out += CHARS.charAt(c1 >> 2);
		out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
		out += CHARS.charAt(((c2 & 0xf) << 2) | ((c3 & 0xc0) >> 6));
		out += CHARS.charAt(c3 & 0x3f);
	}
	return out;
}

function makeBasicAuth(token) {
	//Purpose of this is to convert the token to the authorization header for basic auth
	//Format is Token with a colon at the end then converted to Base64
	return b64Encode(token + ":");
}

/********************************************************/
/********	ArcherAPI_Login
/********************************************************/
function ArcherAPI_Login(callBack) {
	appendDebug("Login");
	/* build url */
	var url = params["archer_webroot"] + params["archer_ws_root"] + params["archer_loginpath"];

	var postBody =
		'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
		'<soap:Body><CreateUserSessionFromInstance xmlns="http://archer-tech.com/webservices/"><userName>' +
		params["archer_username"] +
		"</userName><instanceName>" +
		params["archer_instanceName"] +
		"</instanceName><password>" +
		params["archer_password"] +
		"</password></CreateUserSessionFromInstance></soap:Body></soap:Envelope>";

	var headers = {
		"content-type": "text/xml; charset=utf-8",
	};

	var results = [];

	var loginToArcher = function (url) {
		LogInfo("----------------------------Login Attempt Starting------------------");
		LogInfo("URL1: " + url);
		LogVerbose("POST1: " + postBody);
		//LogVerbose('HEAD1: ' + headers);

		webCall(url, "POST", headers, postBody, function (err, h, b) {
			if (err) {
				LogError("ERR1: " + err);
				//LogVerbose("HEAD1: " + h);
				LogError("RESP1: " + b);
				appendDebug("ERR1");
				callBack(true, null);
			} else {
				//LogInfo("webCall Results: " + b);
				results = b; //This is the result from the webCall
				var doc = xmlStringToXmlDoc(results);
				//Get the value of the the text which is the session token
				sSessionToken = doc.getElementsByTagName("CreateUserSessionFromInstanceResult")[0].childNodes[0].nodeValue;
				LogInfo("sSessionToken: " + sSessionToken);
				if (sSessionToken.length > 5) {
					GetArcherVersion(callBack);
				} else {
					LogError("sSessionToken blank: " + sSessionToken);
					appendDebug("SessionBlank");
					callBack(true, null);
				}
			}
		});
	};

	/* kick off process */
	loginToArcher(url);
}

/********************************************************/
/********	GetArcherVersion
/********************************************************/
var GetArcherVersion = function (callBack) {
	LogInfo("----------------------------GetArcherVersion----------------------------");

	var url = params["archer_webroot"] + params["archer_rest_root"] + params["archer_version"];

	var headers = {
		"Authorization": 'Archer session-id="' + sSessionToken + '"',
		"Content-type": "application/json",
		"Accept": "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
	};

	var postBody = {};

	LogInfo("URL1.1: " + url);
	LogVerbose("BODY1.1: " + jsonToString(postBody));
	//LogInfo('HEAD: ' + headers);

	webCall(url, "GET", headers, jsonToString(postBody), function (err, h, b) {
		if (err) {
			LogSaaSError();
			LogError("ERR1.1: " + err);
			//LogInfo("HERE HEAD1: " + h);
			LogError("BODY1.1: " + b);
		} else {
			LogVerbose("Raw Report Results: " + b);
			var resultJSON = JSON.parse(b);

			if (typeof resultJSON != "undefined" && typeof resultJSON.RequestedObject != "undefined" && resultJSON.RequestedObject.Version != "undefined") {
				sArcherVersionNum = resultJSON.RequestedObject.Version; //Get the Version
				COST_Platform = COST_Platform + " (" + sArcherVersionNum + ")";
				LogInfo("sArcherVersionNum: " + sArcherVersionNum);
				GetArcherBitSightCompanies(callBack);
			} else {
				sArcherVersionNum = "Unknown";
				LogWarn("ERROR Obtaining Archer Version Number: " + b);
				GetArcherBitSightCompanies(callBack);
			}
		}
	});
};

/************************************************************************************************************************************************************************************************************************************/
/********	GetArcherBitSightCompanies
/************************************************************************************************************************************************************************************************************************************/
function GetArcherBitSightCompanies(callBack) {
	/* build url */
	var url = params["archer_webroot"] + params["archer_ws_root"] + params["archer_searchpath"];

	var headers = { "content-type": "text/xml; charset=utf-8", "SOAPAction": "http://archer-tech.com/webservices/ExecuteSearch" };

	//Building a search criteria instead of using the report. This way we can return thousands of records instead of having to page through 250 records at a time...

	//Search criteria to get relevant fields from BitSight Portfolio application where BitSight GUID is not empty
	var sSearchCriteria;
	if (!bIsLegacyIntegration) {
		sSearchCriteria =
			"<SearchReport><PageSize>20000</PageSize><MaxRecordCount>20000</MaxRecordCount><DisplayFields>" +
			'<DisplayField name="BitSight Company Name">d616bf90-7cb7-4858-8f26-fddf66d618c8</DisplayField>' +
			'<DisplayField name="BitSight GUID">ffb06a0b-a5bd-4fff-9250-76f6a757d926</DisplayField>' +
			'<DisplayField name="Botnet Infections">6d0425ec-e509-4022-9a9b-1c3458c2c60e</DisplayField>' +
			'<DisplayField name="Company ID">5b4118e8-e5e6-4103-8511-f87dd7625a87</DisplayField>' +
			'<DisplayField name="Desktop Software">d96b5db0-adf1-4af2-b59c-e846ecfded67</DisplayField>' +
			'<DisplayField name="DKIM Records">8916401c-dede-4659-a258-11d88a3a7fb3</DisplayField>' +
			'<DisplayField name="DNSSEC">b249b1c6-7f4b-48fc-ac41-dc4a66209d7a</DisplayField>' +
			'<DisplayField name="File Sharing">9bad08b0-4332-42a9-8675-66b20aadc775</DisplayField>' +
			'<DisplayField name="Insecure Systems">ddd12adb-349f-4493-b628-ab79d1d86923</DisplayField>' +
			'<DisplayField name="Malware Servers">e3f7356f-5478-4805-a7b7-4ef863b01179</DisplayField>' +
			'<DisplayField name="Mobile Application Security">69bed8f1-d68b-4571-a3b0-04e7409ebdfc</DisplayField>' +
			'<DisplayField name="Mobile Software">f7b8ccb4-80bd-43b6-bf7f-e43f531d4539</DisplayField>' +
			'<DisplayField name="Open Ports">080c4301-0219-42d3-b2c4-6783aa11dc1a</DisplayField>' +
			'<DisplayField name="Patching Cadence">2d8cbd62-f04f-4b72-b1ec-b41c9ac9bbb7</DisplayField>' +
			'<DisplayField name="Potentially Exploited">2c0e1cf5-bd7c-4859-ba54-674df3c76978</DisplayField>' +
			'<DisplayField name="Rating">e6c8ae53-3e49-40f3-b201-3ec18ed661c1</DisplayField>' +
			'<DisplayField name="Rating Category">d9fb6ee5-f8fd-448e-8e18-fe52278bbe8a</DisplayField>' +
			'<DisplayField name="Rating Date">cd34aff5-3f48-47be-a629-87b2579d38a8</DisplayField>' +
			'<DisplayField name="Security Incidents/Breaches">63beaccb-1ef6-40f3-a998-3a5d469095a4</DisplayField>' +
			'<DisplayField name="Server Software">eb0f9046-2a57-4f82-b0e7-936c242b5c63</DisplayField>' +
			'<DisplayField name="Spam Propagation">ae1805d4-2405-4310-8d99-4fddbf624fd8</DisplayField>' +
			'<DisplayField name="SPF Domains">e6eb290b-b986-4ada-b3aa-6f1bfcd6e14e</DisplayField>' +
			'<DisplayField name="Subscription Type">b28ff0da-a9ec-4539-a2f1-7bdcd77a774c</DisplayField>' +
			'<DisplayField name="TLS/SSL Certificates">e37379d7-c383-4efc-8be5-75c0b352073d</DisplayField>' +
			'<DisplayField name="TLS/SSL Configurations">c61e3eec-7ab9-43d0-a1a5-e08b65c1f680</DisplayField>' +
			'<DisplayField name="Tracking ID">1adbe906-14d0-4cee-bd4d-2b87577ccf75</DisplayField>' +
			'<DisplayField name="Unsolicited Communications">395a5113-fab9-4c71-bb12-7b1d60c33dde</DisplayField>' +
			'<DisplayField name="Web Application Headers">99ac94c3-bcdc-4521-ae0c-260d773a1e91</DisplayField>' +
			'<DisplayField name="Tier">' +
			params["archer_bitsightportfolio_tier_guid"] +
			"</DisplayField>" +
			'</DisplayFields><Criteria><ModuleCriteria><Module name="BitSight Portfolio">21d4cfb5-e4d2-4c0b-a34f-559e3a0ef004</Module>' +
			'<SortFields><SortField name="Sort1"><Field name="BitSight Company Name">d616bf90-7cb7-4858-8f26-fddf66d618c8</Field>' +
			'<SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter><Conditions><TextFilterCondition name="Text 1">' +
			'<Field name="BitSight GUID">ffb06a0b-a5bd-4fff-9250-76f6a757d926</Field><Operator>DoesNotEqual</Operator><Value>' +
			"</Value></TextFilterCondition></Conditions></Filter></Criteria></SearchReport>";
	} else {
		sSearchCriteria =
			"<SearchReport><PageSize>20000</PageSize><MaxRecordCount>20000</MaxRecordCount><DisplayFields>" +
			'<DisplayField name="BitSight Company Name">C9ABC083-CC49-498D-8ACA-FD1BEE4C9A5A</DisplayField>' +
			'<DisplayField name="BitSight GUID">6E965C7C-84A7-4A98-9701-2EF0E980D967</DisplayField>' +
			'<DisplayField name="Botnet Infections">2780544A-F720-4C53-B26E-B24C463504AD</DisplayField>' +
			'<DisplayField name="Company ID">9A97EF7F-3B3E-402B-AF71-62B0D8718D9C</DisplayField>' +
			'<DisplayField name="Desktop Software">EDFD28BF-0179-4512-A558-18DD3E647FE5</DisplayField>' +
			'<DisplayField name="DKIM Records">18D0569E-BEDC-409F-A3FD-F13F766D231F</DisplayField>' +
			'<DisplayField name="DNSSEC">EF74BA77-D246-4BF8-A989-5414E38BB963</DisplayField>' +
			'<DisplayField name="File Sharing">08595AAF-B1DF-4FC1-AE8C-9865D3208D4E</DisplayField>' +
			'<DisplayField name="Insecure Systems">888971C3-635F-4322-829F-6BF9674368AA</DisplayField>' +
			'<DisplayField name="Malware Servers">85A29C99-F51A-46F8-93CE-FE632CCD58B3</DisplayField>' +
			'<DisplayField name="Mobile Application Security">897A4772-CA7E-49F5-897F-E20053AEF5A3</DisplayField>' +
			'<DisplayField name="Mobile Software">2C247FA0-5123-45FE-B01D-7F59CD43C733</DisplayField>' +
			'<DisplayField name="Open Ports">20A6C851-DA90-4A9E-9EC3-CB661770C42A</DisplayField>' +
			'<DisplayField name="Patching Cadence">7C5375EA-99BA-445D-B749-0AF1651D267D</DisplayField>' +
			'<DisplayField name="Potentially Exploited">D51BDBE6-12A6-4F00-8671-F95D7588C112</DisplayField>' +
			'<DisplayField name="Rating">4C78DE0A-F8A7-4359-BDAD-57B50A83F05B</DisplayField>' +
			'<DisplayField name="Rating Category">D0BE9FD5-9782-4B6E-B8BD-5BF711A73128</DisplayField>' +
			'<DisplayField name="Rating Date">10344C28-E747-4BB3-91B1-E006B4723436</DisplayField>' +
			'<DisplayField name="Security Incidents/Breaches">76D61FDD-AA99-450D-A5E0-7F2DB14F77A5</DisplayField>' +
			'<DisplayField name="Server Software">067554D6-A46B-480F-B209-585BC51EC8D9</DisplayField>' +
			'<DisplayField name="Spam Propagation">501C2465-181A-4F0D-97A2-D1FB0A896F7F</DisplayField>' +
			'<DisplayField name="SPF Domains">2B9EEB52-8BE8-4F2D-8722-CABA84BB0946</DisplayField>' +
			'<DisplayField name="Subscription Type">D556BA3E-1AED-4E1D-867E-70EFB8B9586C</DisplayField>' +
			'<DisplayField name="TLS/SSL Certificates">C35C3AA0-1A7E-4DEB-A639-43317E74F459</DisplayField>' +
			'<DisplayField name="TLS/SSL Configurations">49FBA42D-DE32-477E-B003-78F7A86B7049</DisplayField>' +
			'<DisplayField name="Tracking ID">BFC34D62-D854-4536-9959-EEC2379886ED</DisplayField>' +
			'<DisplayField name="Unsolicited Communications">C6005E93-CE6E-487B-B3B9-D19DFC6A0B5E</DisplayField>' +
			'<DisplayField name="Web Application Headers">140FA5E0-AD55-4BC6-B260-6287275202CA</DisplayField>' +
			'<DisplayField name="Tier">' +
			params["archer_bitsightportfolio_tier_guid"] +
			"</DisplayField>" +
			'</DisplayFields><Criteria><ModuleCriteria><Module name="BitSight Portfolio">54FE69B0-3C28-4D94-852D-841151634F22</Module>' +
			'<SortFields><SortField name="Sort1"><Field name="BitSight Company Name">C9ABC083-CC49-498D-8ACA-FD1BEE4C9A5A</Field>' +
			'<SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter><Conditions><TextFilterCondition name="Text 1">' +
			'<Field name="BitSight GUID">6E965C7C-84A7-4A98-9701-2EF0E980D967</Field><Operator>DoesNotEqual</Operator><Value>' +
			"</Value></TextFilterCondition></Conditions></Filter></Criteria></SearchReport>";
	}

	LogVerbose("sSearchCriteria: " + sSearchCriteria);

	//Must escape the XML to next inside of the soap request...
	sSearchCriteria = sSearchCriteria.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

	var postBody =
		'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
		'<soap:Body><ExecuteSearch xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
		sSessionToken +
		"</sessionToken>" +
		"<searchOptions>" +
		sSearchCriteria +
		"</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";

	var results = [];

	var archerAPI_GetArcherBitSightCompanies = function (url) {
		LogInfo("----------------------------GET Archer BitSight Companies----------------------------");
		LogInfo("URL2: " + url);
		LogVerbose("BODY2: " + postBody);
		//LogVerbose('HEAD2: ' + headers);

		webCall(url, "POST", headers, postBody, function (err, h, b) {
			if (err) {
				LogError("ERR2: " + err);
				//LogVerbose("HERE HEAD2: " + h);
				LogError("BODY2: " + b);
				callBack(true, null);
			} else {
				//LogVerbose("Raw Report Results: " + b);
				results = b; //This is the result from the webCall
				reviewResults();
			}
		});
	};

	var reviewResults = function () {
		//Convert XML to an XMLDOM
		var doc = xmlStringToXmlDoc(results);

		//Check to see if nothing was returned from the search query
		if (
			typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined"
		) {
			//There weren't any records
			LogInfo("Either no results or Archer search error. Go get active BitSight Subscriptions.");
			GetCurrentSubscriptions(callBack);
		} //Need to proceed and check the count anyway.
		else {
			//Need to get the xml inside the SOAP request and url decode the results
			var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;

			//variable for our json object
			var resultJSON = null;
			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) {
				resultJSON = result; //get the result into the object we can use below;
			});

			if (bVerboseLogging === true) {
				var fs = require("fs");
				fs.writeFileSync("logs\\" + appPrefix + "\\" + "archerBitSightCompanies.json", JSON.stringify(resultJSON));
				LogInfo("Saved to logs\\" + appPrefix + "\\" + "archerBitSightCompanies.json file");
			}

			var iNumCompanies = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("iNumCompanies=" + iNumCompanies);

			//Check to see if we have any existing records
			if (iNumCompanies == 0) {
				LogInfo("No existing Archer BitSight records. Go get active BitSight Subscriptions.");
				GetCurrentSubscriptions(callBack);
			} else {
				compileResults(resultJSON);
			}
		}
	};

	//Results were found, now get the details
	var compileResults = function (resultJSON) {
		var iID_BitSightCompanyGUID;
		var iID_BitSightCompanyName;
		var iID_CompanyID; //Customer driven ID to help with matching
		var iID_application_security; //Web Application Headers?!??!?!?!?
		var iID_botnet_infections;
		var iID_data_breaches;
		var iID_desktop_software;
		var iID_dkim;
		var iID_dnssec;
		var iID_file_sharing;
		var iID_insecure_systems;
		var iID_malware_servers;
		var iID_mobile_application_security;
		var iID_mobile_software;
		var iID_open_ports;
		var iID_patching_cadence;
		var iID_potentially_exploited;
		var iID_rating;
		var iID_rating_date;
		var iID_rating_range;
		var iID_server_software;
		var iID_spam_propagation;
		var iID_spf;
		var iID_ssl_certificates;
		var iID_ssl_configurations;
		var iID_subscripion_type;
		var iID_unsolicited_comm;
		var iID_bitsighttier;

		//Get the LevelID of the Portfolio app
		var sArcherBitSightPortfolio_LevelID = resultJSON.Records.LevelCounts[0].LevelCount[0].$.id;

		//Iterate through the FieldDefinition to get the field ids that we care about
		for (var h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) {
			var sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			var sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;

			//LogInfo("  Alias:" + sAlias + "   ID: " + sID);
			if (sAlias == "BitSight_Company_Name") {
				iID_BitSightCompanyName = sID;
			} else if (sAlias == "BitSight_GUID") {
				iID_BitSightCompanyGUID = sID;
			} else if (sAlias == "Botnet_Infections") {
				iID_botnet_infections = sID;
			} else if (sAlias == "Company_ID") {
				iID_CompanyID = sID;
			} else if (sAlias == "Desktop_Software") {
				iID_desktop_software = sID;
			} else if (sAlias == "DKIM_Records") {
				iID_dkim = sID;
			} else if (sAlias == "DNSSEC") {
				iID_dnssec = sID;
			} else if (sAlias == "File_Sharing") {
				iID_file_sharing = sID;
			} else if (sAlias == "Insecure_Systems") {
				iID_insecure_systems = sID;
			} else if (sAlias == "Malware_Servers") {
				iID_malware_servers = sID;
			} else if (sAlias == "Mobile_Application_Security") {
				iID_mobile_application_security = sID;
			} else if (sAlias == "Mobile_Software") {
				iID_mobile_software = sID;
			} else if (sAlias == "Open_Ports") {
				iID_open_ports = sID;
			} else if (sAlias == "Patching_Cadence") {
				iID_patching_cadence = sID;
			} else if (sAlias == "Potentially_Exploited") {
				iID_potentially_exploited = sID;
			} else if (sAlias == "Rating") {
				iID_rating = sID;
			} else if (sAlias == "Rating_Category") {
				iID_rating_range = sID;
			} else if (sAlias == "Rating_Date") {
				iID_rating_date = sID;
			} else if (sAlias == "Security_Incidents_Breaches") {
				iID_data_breaches = sID;
			} else if (sAlias == "Server_Software") {
				iID_server_software = sID;
			} else if (sAlias == "Spam_Propagation") {
				iID_spam_propagation = sID;
			} else if (sAlias == "SPF_Domains") {
				iID_spf = sID;
			} else if (sAlias == "Subscription_Type") {
				iID_subscripion_type = sID;
			} else if (sAlias == "TLS_SSL_Certificates") {
				iID_ssl_certificates = sID;
			} else if (sAlias == "TLS_SSL_Configurations") {
				iID_ssl_configurations = sID;
			} else if (sAlias == "Unsolicited_Communications") {
				iID_unsolicited_comm = sID;
			} else if (sAlias == "Web_Application_Headers") {
				iID_application_security = sID;
			} else if (sAlias == "BitSight_Tier") {
				iID_bitsighttier = sID;
			}
		}

		var sContentID = "";
		var sBitSightCompanyName;
		var sBitSightCompanyGUID;
		var sCompanyID; //Customer driven ID to help with matching
		var srating_range;
		var srating;
		var srating_date;
		var ssubscripion_type;
		var sbotnet_infections;
		var sspam_propagation;
		var smalware_servers;
		var sunsolicited_comm;
		var spotentially_exploited;
		var sspf;
		var sdkim;
		var sssl_certificates;
		var sssl_configurations;
		var sopen_ports;
		var sapplication_security; //Web Application Headers?!??!?!?!?
		var spatching_cadence;
		var sinsecure_systems;
		var sserver_software;
		var sdesktop_software;
		var smobile_software;
		var sdnssec;
		var smobile_application_security;
		var sfile_sharing;
		var sdata_breaches;
		var sBitSightTier;

		//Iterate through each Archer company record....
		for (var i in resultJSON.Records.Record) {
			LogInfo("-----ARCHER COMPANY RECORD #" + i + "-------");
			sContentID = null;
			sBitSightCompanyName = "";
			sBitSightCompanyGUID = "";
			sCompanyID = ""; //Customer driven ID to help with matching
			srating_range = "";
			srating = "";
			srating_date = "";
			ssubscripion_type = "";
			sbotnet_infections = "";
			sspam_propagation = "";
			smalware_servers = "";
			sunsolicited_comm = "";
			spotentially_exploited = "";
			sspf = "";
			sdkim = "";
			sssl_certificates = "";
			sssl_configurations = "";
			sopen_ports = "";
			sapplication_security = ""; //Web Application Headers?!??!?!?!?
			spatching_cadence = "";
			sinsecure_systems = "";
			sserver_software = "";
			sdesktop_software = "";
			smobile_software = "";
			sdnssec = "";
			smobile_application_security = "";
			sfile_sharing = "";
			sdata_breaches = "";
			sBitSightTier = "";

			//Get the Tracking ID...
			sContentID = resultJSON.Records.Record[i].$.contentId;
			//LogInfo("sContentID: " + sContentID);

			//Iterate through the Field elements for the current config record to get the goodies
			for (var y in resultJSON.Records.Record[i].Field) {
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;

				//Now find all the good data we care about...
				if (sID == iID_BitSightCompanyName) {
					sBitSightCompanyName = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitSightCompanyGUID) {
					sBitSightCompanyGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_CompanyID) {
					sCompanyID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_application_security) {
					sapplication_security = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_botnet_infections) {
					sbotnet_infections = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_data_breaches) {
					sdata_breaches = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_desktop_software) {
					sdesktop_software = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_dkim) {
					sdkim = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_dnssec) {
					sdnssec = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_file_sharing) {
					sfile_sharing = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_insecure_systems) {
					sinsecure_systems = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_malware_servers) {
					smalware_servers = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_mobile_application_security) {
					smobile_application_security = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_mobile_software) {
					smobile_software = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_open_ports) {
					sopen_ports = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_patching_cadence) {
					spatching_cadence = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_potentially_exploited) {
					spotentially_exploited = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_rating) {
					srating = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_rating_date) {
					srating_date = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_rating_range) {
					srating_range = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_server_software) {
					sserver_software = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_spam_propagation) {
					sspam_propagation = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_spf) {
					sspf = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_ssl_certificates) {
					sssl_certificates = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_ssl_configurations) {
					sssl_configurations = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_subscripion_type) {
					ssubscripion_type = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_unsolicited_comm) {
					sunsolicited_comm = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_bitsighttier) {
					sBitSightTier = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				}
			}

			LogInfo("Content ID: " + sContentID + " CompanyName:" + sBitSightCompanyName);
			//Populate the main record with the details we care about....
			aBitSightRatings[aBitSightRatings.length] = {
				"ContentID": sContentID,
				"CompanyName": sBitSightCompanyName,
				"CompanyID": sCompanyID,
				"GUID": sBitSightCompanyGUID,
				"subscription_type": ssubscripion_type,
				"rating_date": srating_date,
				"rating": srating,
				"range": srating_range,
				"botnet_infections": sbotnet_infections,
				"spam_propagation": sspam_propagation,
				"malware_servers": smalware_servers,
				"unsolicited_comm": sunsolicited_comm,
				"potentially_exploited": spotentially_exploited,
				"spf": sspf,
				"dkim": sdkim,
				"ssl_certificates": sssl_certificates,
				"ssl_configurations": sssl_configurations,
				"open_ports": sopen_ports,
				"application_security": sapplication_security,
				"patching_cadence": spatching_cadence,
				"insecure_systems": sinsecure_systems,
				"server_software": sserver_software,
				"desktop_software": sdesktop_software,
				"mobile_software": smobile_software,
				"dnssec": sdnssec,
				"mobile_application_security": smobile_application_security,
				"file_sharing": sfile_sharing,
				"data_breaches": sdata_breaches,
				"BitSightTier": null,
			};

			LogInfo("*Company Number#" + aBitSightRatings.length + "=" + JSON.stringify(aBitSightRatings[aBitSightRatings.length - 1]));
		}

		//Just for testing...save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "aArcherBitSightRatings.json", JSON.stringify(aBitSightRatings));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aArcherBitSightRatings.json file");
		}

		//Now that we have the current info, now go get the info from the BitSight API
		GetCurrentBitSightCompanyInfo(callBack);
	};

	/* kick off process */
	archerAPI_GetArcherBitSightCompanies(url);
}

/************************************************************************************************************************************************************************************************************************************/
/********	GetCurrentBitSightCompanyInfo
/************************************************************************************************************************************************************************************************************************************/
function GetCurrentBitSightCompanyInfo(callBack) {
	var aBitSightCompanyInfo;
	var bitSightAPI_GetCurrentBitSightCompanyInfo = function (callBack) {
		var url = params["bitsight_webroot"] + params["bitsight_currentuser"];

		var sAuthorization = makeBasicAuth(params["bitsight_token"]);

		var headers = { "Accept": "application/json", "Authorization": "Basic " + sAuthorization };

		var postBody = null;

		LogInfo("----------------------GetCurrentBitSightCompanyInfo----------------------");
		LogInfo("URL3: " + url);
		//LogVerbose('BODY3: ' + JSON.stringify(postBody));
		//LogInfo('HEAD3: ' + JSON.stringify(headers));

		/* build options */
		var options = {
			method: "GET",
			uri: url,
			headers: headers,
			body: postBody,
			timeout: httpTimeout,
			rejectUnauthorized: verifyCerts,
		};

		/* make the request */
		request(options, function (err1, response1, body) {
			var errMessage = null;

			/* check for error */
			if (err1) {
				errMessage = "HTTP ERROR: " + err1;
			} else {
				//LogInfo("Header Response: " + response1.headers);
				if (response1.statusCode != 200) {
					errMessage = "HTTP " + response1.statusCode + " ERROR: " + err1;
				}
			}
			/* check for error */
			if (errMessage) {
				LogError("ERR3: " + errMessage);
				LogError("BODY3: " + body);
				callBack(true, null);
			} else {
				LogInfo("body:" + body);
				aBitSightCompanyInfo = JSON.parse(body);
				LogInfo("----------------------GOT Company Info----------------------");

				//Just for testing...save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightCompanyInfo.json", JSON.stringify(aBitSightCompanyInfo));
					LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightCompanyInfo.json file");
				}

				if (typeof aBitSightCompanyInfo == "undefined" || typeof aBitSightCompanyInfo.customer == "undefined" || typeof aBitSightCompanyInfo.customer.name == "undefined") {
					BitSightCustomerName = "unknown";
				} else {
					BitSightCustomerName = aBitSightCompanyInfo.customer.name;
					LogInfo("BitSight Customer: " + BitSightCustomerName);
				}

				GetCurrentSubscriptions(callBack);
			}
		});
	};

	/* kick off process */
	bitSightAPI_GetCurrentBitSightCompanyInfo(callBack);
}

/************************************************************************************************************************************************************************************************************************************/
/********	GetCurrentSubscriptions
/************************************************************************************************************************************************************************************************************************************/
function GetCurrentSubscriptions(callBack) {
	var bitSightAPI_GetSubscriptions = function (callBack) {
		var url = params["bitsight_webroot"] + params["bitsight_subscriptions"];

		var sAuthorization = makeBasicAuth(params["bitsight_token"]);

		var headers = {
			"Accept": "application/json",
			"Authorization": "Basic " + sAuthorization,
			"X-BITSIGHT-CONNECTOR-NAME-VERSION": COST_ArcherBitSightVersion,
			"X-BITSIGHT-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-BITSIGHT-CUSTOMER": BitSightCustomerName,
		};

		var postBody = null;

		LogInfo("----------------------GET BITSIGHT SUBSCRIPTIONS----------------------");
		LogInfo("URL4: " + url);
		LogInfo("HEAD4: " + JSON.stringify(headers));

		/* build options */
		var options = {
			method: "GET",
			uri: url,
			headers: headers,
			body: postBody,
			timeout: httpTimeout,
			rejectUnauthorized: verifyCerts,
		};

		/* make the request */
		request(options, function (err1, response1, body) {
			var errMessage = null;

			/* check for error */
			if (err1) {
				errMessage = "HTTP ERROR: " + err1;
			} else {
				//LogInfo("Header Response: " + response1.headers);
				if (response1.statusCode != 200) {
					errMessage = "HTTP " + response1.statusCode + " ERROR: " + err1;
				}
			}
			/* check for error */
			if (errMessage) {
				LogError("ERR4: " + errMessage);
				LogError("BODY4: " + body);
				callBack(true, null);
			} else {
				LogInfo("body:" + body);
				aBitSightSubscriptions = JSON.parse(body);
				LogInfo("----------------------GOT Subscriptions----------------------");

				//Just for testing...save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightSubscriptions.json", JSON.stringify(aBitSightSubscriptions));
					LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightSubscriptions.json file");
				}

				ParseSubscriptions(callBack);
			}
		});
	};

	function ParseSubscriptions(callBack) {
		var aBitSightCompanies = [];

		LogInfo("Iterating Total Risk Monitoring Companies");
		//8/6/2024(DC) Added error handling since continuous_monitoring or companies may be empty - when originally developed the JSON block existed whether data existed or it was empty.
		try {
			if (typeof aBitSightSubscriptions["continuous_monitoring"] == "undefined" || typeof aBitSightSubscriptions["continuous_monitoring"].companies == "undefined") {
				LogInfo("Risk Monitoring (alerts-only) Companies were empty, so skipping block.");
			} else {
				for (var i in aBitSightSubscriptions.continuous_monitoring.companies) {
					var sCompanyName = aBitSightSubscriptions.continuous_monitoring.companies[i].name;
					var sGUID = aBitSightSubscriptions.continuous_monitoring.companies[i].guid;
					LogInfo(i.toString() + " " + sCompanyName + ":" + sGUID);

					aBitSightCompanies[aBitSightCompanies.length] = {
						"CompanyName": sCompanyName,
						"GUID": sGUID,
					};
				}
			}
		} catch (ex) {
			LogInfo("GetCurrentSubscriptions() Error parsing continuous_monitoring data. ex: " + ex);
		}

		LogInfo("Iterating Risk Monitoring (alerts-only) Companies");
		//8/6/2024(DC) Added error handling since alert-only or companies may be empty - when originally developed the JSON block existed whether data existed or it was empty.
		try {
			if (typeof aBitSightSubscriptions["alerts-only"] == "undefined" || typeof aBitSightSubscriptions["alerts-only"].companies == "undefined") {
				LogInfo("Risk Monitoring (alerts-only) Companies were empty, so skipping block.");
			} else {
				for (var i in aBitSightSubscriptions["alerts-only"].companies) {
					//Since there's a special character need to access the element this way.
					var sCompanyName = aBitSightSubscriptions["alerts-only"].companies[i].name;
					var sGUID = aBitSightSubscriptions["alerts-only"].companies[i].guid;
					LogInfo(i.toString() + " " + sCompanyName + ":" + sGUID);

					aBitSightCompanies[aBitSightCompanies.length] = {
						"CompanyName": sCompanyName,
						"GUID": sGUID,
					};
				}
			}
		} catch (ex) {
			LogInfo("GetCurrentSubscriptions() Error parsing alerts-only data. ex: " + ex);
		}

		//Now that we have the current subscriptions, let's consolidate that list with what we had from Archer already
		LogInfo("Starting consolidation...");
		//If nothing came from Archer, just set the array to what we got from BitSight.
		if (aBitSightRatings.length == 0) {
			aBitSightRatings = aBitSightCompanies;
		} //otherwise iterate through and find/resolve matches.
		else {
			//Loop through the Companies found in BitSight API
			for (var j in aBitSightCompanies) {
				var sBitSightGUID = aBitSightCompanies[j].GUID;

				var bFoundGUID = false;
				//Loop through the Companies found in Archer already
				for (var i in aBitSightRatings) {
					var sArcherGUID = aBitSightRatings[i].GUID;

					//If there's a match flag it so we can skip adding this record later.
					if (sArcherGUID == sBitSightGUID) {
						bFoundGUID = true;
					}
				}

				//Check if the GUID was found. If not, add it to the array
				if (bFoundGUID == false) {
					aBitSightRatings[aBitSightRatings.length] = {
						"CompanyName": aBitSightCompanies[j].CompanyName,
						"GUID": aBitSightCompanies[j].GUID,
					};
				}
			}
		}
		LogInfo("Completed consolidation.");

		//Just for testing...save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightRatingsCondolidated.json", JSON.stringify(aBitSightRatings));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightRatingsCondolidated.json file");
		}

		GetRatings(0, callBack);
	}

	/* kick off process */
	bitSightAPI_GetSubscriptions(callBack);
}

/************************************************************************************************************************************************************************************************************************************/
/********	GetRatings
/************************************************************************************************************************************************************************************************************************************/
function GetRatings(iCurrentCompany, callBack) {
	LogInfo("----------------------------GetRatings----------------------------");
	LogInfo("STARTING ON Company #: " + iCurrentCompany + " COMPANY: " + aBitSightRatings[iCurrentCompany].CompanyName);
	var url = params["bitsight_webroot"] + params["bitsight_companyratings"] + "/" + aBitSightRatings[iCurrentCompany].GUID;

	var sAuthorization = makeBasicAuth(params["bitsight_token"]);

	var headers = {
		"Accept": "application/json",
		"Authorization": "Basic " + sAuthorization,
		"X-BITSIGHT-CONNECTOR-NAME-VERSION": COST_ArcherBitSightVersion,
		"X-BITSIGHT-CALLING-PLATFORM-VERSION": COST_Platform,
		"X-BITSIGHT-CUSTOMER": BitSightCustomerName,
	};

	var postBody = "";

	LogInfo("URL5: " + url);
	//LogVerbose('BODY5: ' + JSON.stringify(postBody));

	/* build options */
	var options = {
		method: "GET",
		uri: url,
		headers: headers,
		body: postBody,
		timeout: httpTimeout,
		rejectUnauthorized: verifyCerts,
	};

	var ratingsResponse;

	/* make the request */
	request(options, function (err1, response, body) {
		var errMessage = null;
		var bUnsubcribed = false;

		/* check for error */
		if (err1) {
			errMessage = "HTTP ERROR: " + err1;
		} else {
			//LogInfo("Header Response: " + response.headers);
			if (response.statusCode == 403) {
				//Error 403 is when the subscription is no longer available. Need to identifiy this and set the status in the return results.
				LogInfo("403 returned - Details: Unsubscribed");
				bUnsubcribed = true;
			} else if (response.statusCode == 404) {
				//Error 404 is when the company is not found. This typically happens when a new company is requested, but not added to the BitSight library yet. (We have the guid, but it's not live yet)
				bUnsubcribed = true;
				LogInfo("404 returned - Details: This could be a new company not yet in BitSight. Response: " + body);
			} else if (response.statusCode != 200) {
				errMessage = "HTTP " + response.statusCode + " ERROR: " + err1;
			}
		}
		/* check for error */
		if (errMessage) {
			//If skipping when an error is true, then skip over it and log the warning. Otherwise, terminate the entire process.
			if (bSkipCompanyIfError) {
				LogWarn("bSkipCompanyIfError is set to true.");
				LogWarn("ERR5: " + errMessage);
				LogWarn("BODY5: " + body);

				//Iterate through if there were multiple companies
				//Did we hit the max?
				LogInfo("Current=" + iCurrentCompany + " Max=" + aBitSightRatings.length);
				if (iCurrentCompany >= aBitSightRatings.length - 1) {
					LogInfo("Hit maxResults of " + aBitSightRatings.length);
					//Just for testing...save to file...
					if (bVerboseLogging === true) {
						var fs = require("fs");
						fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightRatings.json", JSON.stringify(aBitSightRatings));
						LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightRatings.json file");
					}

					DetermineExportPath(callBack);
				} else {
					//Still have more companies to iterate through...
					iCurrentCompany++; //Increment before running again
					LogInfo("Iterating through next company=" + iCurrentCompany);
					GetRatings(iCurrentCompany, callBack);
				}
			} else {
				LogError("ERR5: " + errMessage);
				LogError("BODY5: " + body);
				callBack(true, null);
			}
		} else {
			LogInfo("----------------------GOT GetRatings----------------------");
			LogVerbose("Raw Results body: " + body);
			ratingsResponse = JSON.parse(body);

			//Check to see if this company was unsubscribed
			if (bUnsubcribed) {
				LogVerbose("Set to Not Subscribed");
				aBitSightRatings[iCurrentCompany].subscription_type = "Not Subscribed";
			} else {
				if (typeof ratingsResponse == "undefined" || typeof ratingsResponse.rating_details == "undefined") {
					//Nothing returned for this company?
					LogWarn("Nothing was returned for this company. Response or rating_details was undefined.");
				} else {
					LogInfo("Extracting ratings/info...");
					var na = "N/A";

					//5/20/2024 - Obtain the Company ID
					if (typeof ratingsResponse.custom_id == "undefined" || ratingsResponse.custom_id == null || ratingsResponse.custom_id == "") {
						aBitSightRatings[iCurrentCompany].custom_id = "";
					} else {
						aBitSightRatings[iCurrentCompany].custom_id = ratingsResponse.custom_id;
					}

					if (typeof ratingsResponse.subscription_type == "undefined") {
						aBitSightRatings[iCurrentCompany].subscription_type = "";
					} else {
						aBitSightRatings[iCurrentCompany].subscription_type = ratingsResponse.subscription_type;
					}

					if (typeof ratingsResponse.description == "undefined") {
						aBitSightRatings[iCurrentCompany].description = "";
					} else {
						aBitSightRatings[iCurrentCompany].description = ratingsResponse.description;
					}

					if (typeof ratingsResponse.primary_domain == "undefined") {
						aBitSightRatings[iCurrentCompany].primary_domain = "";
					} else {
						aBitSightRatings[iCurrentCompany].primary_domain = ratingsResponse.primary_domain;
					}

					if (typeof ratingsResponse.industry == "undefined") {
						aBitSightRatings[iCurrentCompany].industry = "";
					} else {
						aBitSightRatings[iCurrentCompany].industry = ratingsResponse.industry;
					}

					if (
						typeof ratingsResponse.ratings[0] == "undefined" ||
						typeof ratingsResponse.ratings[0].rating_date == "undefined" ||
						typeof ratingsResponse.ratings[0].rating == "undefined" ||
						typeof ratingsResponse.ratings[0].range == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].rating_date = "";
						aBitSightRatings[iCurrentCompany].rating = "";
						aBitSightRatings[iCurrentCompany].range = "";
					} else {
						aBitSightRatings[iCurrentCompany].rating_date = ratingsResponse.ratings[0].rating_date;
						aBitSightRatings[iCurrentCompany].rating = ratingsResponse.ratings[0].rating;
						aBitSightRatings[iCurrentCompany].range = ratingsResponse.ratings[0].range;
					}

					//Detailed Ratings
					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.botnet_infections == "undefined" ||
						typeof ratingsResponse.rating_details.botnet_infections.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].botnet_infections = na;
					} else {
						aBitSightRatings[iCurrentCompany].botnet_infections = ratingsResponse.rating_details.botnet_infections.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.spam_propagation == "undefined" ||
						typeof ratingsResponse.rating_details.spam_propagation.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].spam_propagation = na;
					} else {
						aBitSightRatings[iCurrentCompany].spam_propagation = ratingsResponse.rating_details.spam_propagation.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.malware_servers == "undefined" ||
						typeof ratingsResponse.rating_details.malware_servers.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].malware_servers = na;
					} else {
						aBitSightRatings[iCurrentCompany].malware_servers = ratingsResponse.rating_details.malware_servers.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.unsolicited_comm == "undefined" ||
						typeof ratingsResponse.rating_details.unsolicited_comm.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].unsolicited_comm = na;
					} else {
						aBitSightRatings[iCurrentCompany].unsolicited_comm = ratingsResponse.rating_details.unsolicited_comm.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.potentially_exploited == "undefined" ||
						typeof ratingsResponse.rating_details.potentially_exploited.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].potentially_exploited = na;
					} else {
						aBitSightRatings[iCurrentCompany].potentially_exploited = ratingsResponse.rating_details.potentially_exploited.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.spf == "undefined" ||
						typeof ratingsResponse.rating_details.spf.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].spf = na;
					} else {
						aBitSightRatings[iCurrentCompany].spf = ratingsResponse.rating_details.spf.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.dkim == "undefined" ||
						typeof ratingsResponse.rating_details.dkim.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].dkim = na;
					} else {
						aBitSightRatings[iCurrentCompany].dkim = ratingsResponse.rating_details.dkim.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.ssl_certificates == "undefined" ||
						typeof ratingsResponse.rating_details.ssl_certificates.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].ssl_certificates = na;
					} else {
						aBitSightRatings[iCurrentCompany].ssl_certificates = ratingsResponse.rating_details.ssl_certificates.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.ssl_configurations == "undefined" ||
						typeof ratingsResponse.rating_details.ssl_configurations.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].ssl_configurations = na;
					} else {
						aBitSightRatings[iCurrentCompany].ssl_configurations = ratingsResponse.rating_details.ssl_configurations.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.open_ports == "undefined" ||
						typeof ratingsResponse.rating_details.open_ports.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].open_ports = na;
					} else {
						aBitSightRatings[iCurrentCompany].open_ports = ratingsResponse.rating_details.open_ports.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.application_security == "undefined" ||
						typeof ratingsResponse.rating_details.application_security.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].application_security = na;
					} else {
						aBitSightRatings[iCurrentCompany].application_security = ratingsResponse.rating_details.application_security.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.patching_cadence == "undefined" ||
						typeof ratingsResponse.rating_details.patching_cadence.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].patching_cadence = na;
					} else {
						aBitSightRatings[iCurrentCompany].patching_cadence = ratingsResponse.rating_details.patching_cadence.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.insecure_systems == "undefined" ||
						typeof ratingsResponse.rating_details.insecure_systems.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].insecure_systems = na;
					} else {
						aBitSightRatings[iCurrentCompany].insecure_systems = ratingsResponse.rating_details.insecure_systems.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.server_software == "undefined" ||
						typeof ratingsResponse.rating_details.server_software.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].server_software = na;
					} else {
						aBitSightRatings[iCurrentCompany].server_software = ratingsResponse.rating_details.server_software.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.desktop_software == "undefined" ||
						typeof ratingsResponse.rating_details.desktop_software.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].desktop_software = na;
					} else {
						aBitSightRatings[iCurrentCompany].desktop_software = ratingsResponse.rating_details.desktop_software.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.mobile_software == "undefined" ||
						typeof ratingsResponse.rating_details.mobile_software.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].mobile_software = na;
					} else {
						aBitSightRatings[iCurrentCompany].mobile_software = ratingsResponse.rating_details.mobile_software.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.dnssec == "undefined" ||
						typeof ratingsResponse.rating_details.dnssec.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].dnssec = na;
					} else {
						aBitSightRatings[iCurrentCompany].dnssec = ratingsResponse.rating_details.dnssec.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.mobile_application_security == "undefined" ||
						typeof ratingsResponse.rating_details.mobile_application_security.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].mobile_application_security = na;
					} else {
						aBitSightRatings[iCurrentCompany].mobile_application_security = ratingsResponse.rating_details.mobile_application_security.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.file_sharing == "undefined" ||
						typeof ratingsResponse.rating_details.file_sharing.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].file_sharing = na;
					} else {
						aBitSightRatings[iCurrentCompany].file_sharing = ratingsResponse.rating_details.file_sharing.grade;
					}

					if (
						typeof ratingsResponse.rating_details == "undefined" ||
						ratingsResponse.rating_details == null ||
						typeof ratingsResponse.rating_details.data_breaches == "undefined" ||
						typeof ratingsResponse.rating_details.data_breaches.grade == "undefined"
					) {
						aBitSightRatings[iCurrentCompany].data_breaches = na;
					} else {
						aBitSightRatings[iCurrentCompany].data_breaches = ratingsResponse.rating_details.data_breaches.grade;
					}
				}
			}

			//Iterate through if there were multiple companies
			//Did we hit the max?
			LogInfo("Current=" + iCurrentCompany + " Max=" + aBitSightRatings.length);
			if (iCurrentCompany >= aBitSightRatings.length - 1) {
				LogInfo("Hit maxResults of " + aBitSightRatings.length);
				//Just for testing...save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightRatings.json", JSON.stringify(aBitSightRatings));
					LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightRatings.json file");
				}

				//DetermineExportPath(callBack); //2/22/2025 update to add Tier
				GetBitSightTiers(callBack);
			} else {
				//Still have more companies to iterate through...
				iCurrentCompany++; //Increment before running again
				LogInfo("Iterating through next company=" + iCurrentCompany);
				GetRatings(iCurrentCompany, callBack);
			}
		}
	});
}

/************************************************************************************************************************************************************************************************************************************/
/********	GetBitSightTiers - Added 2/22/2025 to get the Tiers of the companies
/************************************************************************************************************************************************************************************************************************************/
function GetBitSightTiers(callBack) {
	var bitSightAPI_GetBitSightTiers = function (callBack) {
		var url = params["bitsight_webroot"] + params["bitsight_tiers"];

		var sAuthorization = makeBasicAuth(params["bitsight_token"]);

		var headers = { "Accept": "application/json", "Authorization": "Basic " + sAuthorization };

		var postBody = null;

		LogInfo("----------------------GetBitSightTiers----------------------");
		LogInfo("URL5.1: " + url);
		//LogVerbose('BODY3: ' + JSON.stringify(postBody));
		//LogInfo('HEAD3: ' + JSON.stringify(headers));

		/* build options */
		var options = {
			method: "GET",
			uri: url,
			headers: headers,
			body: postBody,
			timeout: httpTimeout,
			rejectUnauthorized: verifyCerts,
		};

		/* make the request */
		request(options, function (err1, response1, body) {
			var errMessage = null;

			/* check for error */
			if (err1) {
				errMessage = "HTTP ERROR: " + err1;
			} else {
				//LogInfo("Header Response: " + response1.headers);
				if (response1.statusCode != 200) {
					errMessage = "HTTP " + response1.statusCode + " ERROR: " + err1;
				}
			}
			/* check for error */
			if (errMessage) {
				//LogError("ERR5.1: " + errMessage);
				//LogError("BODY5.1: " + body);
				//callBack(true, null);
				LogInfo("GetBitSightTiers() Tier API issue unavailable. skipping tier data.");
				DetermineExportPath(callBack);
			} else {
				LogInfo("body:" + body);
				aBitSightTierInfo = JSON.parse(body);
				LogInfo("----------------------GOT Tier Info----------------------");

				//Just for testing...save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightTierInfo.json", JSON.stringify(aBitSightTierInfo));
					LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightTierInfo.json file");
				}

				if (typeof aBitSightTierInfo == "undefined" || aBitSightTierInfo == null || aBitSightTierInfo.length == 0) {
					LogInfo("GetBitSightTiers() Tiers unavailable. aBitSightTierInfo empty.");
					DetermineExportPath(callBack);
				} else {
					LogInfo("GetBitSightTiers() Tiers retrieved.");
					parse_BitSightTiers(aBitSightTierInfo, callBack);
				}
			}
		});
	};

	var parse_BitSightTiers = function (aBitSightTierInfo, callBack) {
		//Tier Names are configurable by the customer in BitSight. As of 2/22/2025, up to 5 tiers are available.
		//Default values are Tier 1, Tier 2, and Tier 3
		//Archer values must match exactly to what is in BitSight or the data will be empty.

		//Goals:
		// 1. Add Tier Name value to the company object
		// 2. Add to generic object with tier names so we can match to Archer before updating Archer in the GetBitSightTierListValue() function

		//Loop the Tiers object from BitSight
		for (let t in aBitSightTierInfo) {
			let sTierName = aBitSightTierInfo[t].name;
			//let sTierGUID = aBitSightTierInfo[t].guid; //May consider using GUID so just holding on to this line.

			//Loop the companies in this Tier
			//Note there may not be any companies in this tier.
			for (let c in aBitSightTierInfo[t].companies) {
				let sCompanyGUID = aBitSightTierInfo[t].companies[c];

				//Now loop through aBitSightRatings to find match in our BitSight Companies object so we can add the data to it.
				for (let bCompany in aBitSightRatings) {
					if (sCompanyGUID == aBitSightRatings[bCompany].GUID) {
						//Match found...add Tier Name to the aBitSightRatings object for this company
						aBitSightRatings[bCompany].BitSightTier = sTierName;
					}
				}
			}
		}

		//Just for testing...save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightRatings2.json", JSON.stringify(aBitSightRatings));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightRatings2.json file");
		}

		//Continue execution to the next steps...
		DetermineExportPath(callBack);
	};

	/* kick off process */
	bitSightAPI_GetBitSightTiers(callBack);
}

function DetermineExportPath(callBack) {
	if (params["bIsSaaS"] == "true") {
		//This functionality is only for SaaS/Hosted clients running this script outside of Archer (Node.js Prompt).
		if (aBitSightRatings.length == 0) {
			LogInfo("Nothing to send back to Archer...exiting.");
			LogSaaSSuccess();
		} else {
			getPortfolioFieldIDs(callBack);
		}
	} else {
		//Return data back to Archer data feed
		LogInfo("Non-SaaS - returning data to Archer.");
		ReturnDataToArcher(callBack);
	}
}

/********************************************************/
/********	getPortfolioFieldIDs
/********************************************************/
var getPortfolioFieldIDs = function (callBack) {
	LogInfo("----------------------------getPortfolioFieldIDs----------------------------");
	var getPortfolioModuleID = function (callBack) {
		LogInfo("----------------------------getPortfolioAppID----------------------------");
		var url = params["archer_webroot"] + params["archer_rest_root"] + params["archer_applicationpath"];

		var headers = {
			"Authorization": 'Archer session-id="' + sSessionToken + '"',
			"Content-type": "application/json",
			"Accept": "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
		};

		var odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params["archer_BitSightPortfolioApp"] + "'";
		var postBody = {
			"Value": odataquery,
		};

		LogInfo("URL6: " + url);
		LogVerbose("BODY6: " + jsonToString(postBody));

		webCall(url, "GET", headers, jsonToString(postBody), function (err, h, b) {
			if (err) {
				LogError("ERR6: " + err);
				//LogVerbose("HEAD6: " + h);
				LogError("BODY6: " + b);
				callBack(true, null);
			} else {
				LogVerbose("Raw Results: " + b);
				var resultJSON = JSON.parse(b);

				if (typeof resultJSON != "undefined" && typeof resultJSON[0].RequestedObject != "undefined" && resultJSON[0].RequestedObject.Id != "undefined") {
					var iModuleID = resultJSON[0].RequestedObject.Id; //Get the content ID
					LogInfo("iModuleID: " + iModuleID);
					getFieldIDs(callBack, iModuleID);
				} else {
					LogError("ERROR Obtaining BitSight Portfolio module ID: " + b);
				}
			}
		});
	};

	var getFieldIDs = function (callBack, iModuleID) {
		LogInfo("----------------------------getPortfolioFieldIDs----------------------------");
		var url = params["archer_webroot"] + params["archer_rest_root"] + params["archer_fielddefinitionapppath"] + iModuleID;

		var headers = {
			"Authorization": 'Archer session-id="' + sSessionToken + '"',
			"Content-type": "application/json",
			"Accept": "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
		};

		var postBody = {
			"Value": "?$orderby=Name",
		};

		LogInfo("URL7: " + url);
		LogVerbose("BODY7: " + jsonToString(postBody));
		//LogVerbose('HEAD: ' + headers);

		webCall(url, "GET", headers, jsonToString(postBody), function (err, h, b) {
			if (err) {
				LogError("ERR7: " + err);
				//LogVerbose("HEAD7: " + h);
				LogError("BODY7: " + b);
				callBack(true, null);
			} else {
				LogVerbose("Portfolio App Definition: " + b);
				var resultJSON = JSON.parse(b);

				if (typeof resultJSON != "undefined" && typeof resultJSON[0].RequestedObject != "undefined") {
					LogVerbose("Archer Returned good app data");

					//Thankfully LevelID is included in every field and since Portolio is not a leveled app, we can just grab it off the first field
					SaaSParams["LevelID"] = resultJSON[0].RequestedObject.LevelId;

					for (var iField in resultJSON) {
						var sfieldname = resultJSON[iField].RequestedObject.Name.toLowerCase();
						var salias = resultJSON[iField].RequestedObject.Alias.toLowerCase();
						var sguid = resultJSON[iField].RequestedObject.Guid.toLowerCase();

						//LogVerbose("*Looking for: " + sfieldname);

						//Check field name, alias, and guid since client might rename one or more items and GUID could theoretically be used in another app
						if (sfieldname == "bitsight company name" || salias == "bitsight_company_name" || sguid == "d616bf90-7cb7-4858-8f26-fddf66d618c8") {
							//LogVerbose("Got CompanyName");
							SaaSParams["CompanyName"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "bitsight guid" || salias == "bitsight_guid" || sguid == "ffb06a0b-a5bd-4fff-9250-76f6a757d926") {
							SaaSParams["GUID"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "change subscription type" || salias == "change_subscription_type" || sguid == "52465d77-89f4-449b-b38b-a9f5bcf854a1") {
							SaaSParams["changesubscriptionType"] = resultJSON[iField].RequestedObject.Id;

							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "changesubscriptionType",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "changesubscriptionType_",
							};
						} else if (sfieldname == "company id" || salias == "company_id" || sguid == "5b4118e8-e5e6-4103-8511-f87dd7625a87") {
							SaaSParams["company_id"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "current subscription type" || salias == "current_subscription_type" || sguid == "b28ff0da-a9ec-4539-a2f1-7bdcd77a774c") {
							SaaSParams["subscripion_type"] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "subscripion_type",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "subscriptionType_",
							};
						} else if (sfieldname == "description" || salias == "description" || sguid == "2dad4202-95f7-424f-bd9c-b2e2dd15ea32") {
							SaaSParams["description"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "domain" || salias == "domain" || sguid == "a3a80d7d-8fa9-4648-86e7-288380219ce0") {
							SaaSParams["domain"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "headline ratings expiration date" || salias == "headline_ratings_expiration_date" || sguid == "f54966ac-1337-402e-8fb2-cfa0f6fe49b6") {
							SaaSParams["headlineratingexpirationdate"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "industry" || salias == "industry" || sguid == "425f8d2d-fd24-4013-b9e2-ca904514c63f") {
							SaaSParams["industry"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "last risk vectors update" || salias == "last_total_risk_monitoring_update" || sguid == "62e6d1ba-8181-45f7-abb2-6adec774ecf2") {
							SaaSParams["last_total_risk_monitoring_update"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "license bitsight headline ratings?" || salias == "license_1_year_of_bitsight_ratings" || sguid == "18b5a260-2447-4b10-a5b6-22980b5f4415") {
							SaaSParams["licence1yearofratings"] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "licence1yearofratings",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "licence1yearofratings_",
							};
						} else if (sfieldname == "pull today's bitsight risk vectors?" || salias == "pull_todays_bitsight_risk_vectors" || sguid == "ce47490e-aaca-4aed-a50f-332ec4ff210f") {
							SaaSParams["pulltodaysbitsightriskvectors"] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "subscripion_type",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "pulltodaysbitsightriskvectors_",
							};
						} else if (sfieldname == "latest rating" || salias == "rating" || sguid == "e6c8ae53-3e49-40f3-b201-3ec18ed661c1") {
							SaaSParams["rating"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "rating date" || salias == "rating_date" || sguid == "cd34aff5-3f48-47be-a629-87b2579d38a8") {
							SaaSParams["rating_date"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "rating category" || salias == "rating_category" || sguid == "d9fb6ee5-f8fd-448e-8e18-fe52278bbe8a") {
							SaaSParams["rating_range"] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "rating_range",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "range_",
							};
						} else if (sfieldname == "related bitsight alerts" || salias == "related_bitsight_alerts" || sguid == "af683a4c-0e3d-45d8-8b3b-bdd0b6a7a7ba") {
							SaaSParams["related_bitsight_alerts"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "related third party profile" || salias == "related_third_party_profile" || sguid == "8958bdaa-d270-4418-a7f0-8846b882f0d3") {
							SaaSParams["RelatedTPP"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "subscription change results" || salias == "subscription_change_results" || sguid == "c97fdd37-8dc1-4584-a329-996fc5d1b554") {
							SaaSParams["subscriptionchangeresults"] = resultJSON[iField].RequestedObject.Id;
						}

						//Vectors
						else if (sfieldname == "botnet infections" || salias == "botnet_infections" || sguid == "6d0425ec-e509-4022-9a9b-1c3458c2c60e") {
							SaaSParams["botnet_infections"] = resultJSON[iField].RequestedObject.Id;
							//Only need to get this once since it's a global values list
							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "rating",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "rating_",
							};
						} else if (sfieldname == "desktop software" || salias == "desktop_software" || sguid == "d96b5db0-adf1-4af2-b59c-e846ecfded67") {
							SaaSParams["desktop_software"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "dkim records" || salias == "dkim_records" || sguid == "8916401c-dede-4659-a258-11d88a3a7fb3") {
							SaaSParams["dkim"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "dnssec" || salias == "dnssec" || sguid == "b249b1c6-7f4b-48fc-ac41-dc4a66209d7a") {
							SaaSParams["dnssec"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "file sharing" || salias == "file_sharing" || sguid == "9bad08b0-4332-42a9-8675-66b20aadc775") {
							SaaSParams["file_sharing"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "insecure systems" || salias == "insecure_systems" || sguid == "ddd12adb-349f-4493-b628-ab79d1d86923") {
							SaaSParams["insecure_systems"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "malware servers" || salias == "malware_servers" || sguid == "e3f7356f-5478-4805-a7b7-4ef863b01179") {
							SaaSParams["malware_servers"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "mobile application security" || salias == "mobile_application_security" || sguid == "69bed8f1-d68b-4571-a3b0-04e7409ebdfc") {
							SaaSParams["mobile_application_security"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "mobile software" || salias == "mobile_software" || sguid == "f7b8ccb4-80bd-43b6-bf7f-e43f531d4539") {
							SaaSParams["mobile_software"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "open ports" || salias == "open_ports" || sguid == "080c4301-0219-42d3-b2c4-6783aa11dc1a") {
							SaaSParams["open_ports"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "patching cadence" || salias == "patching_cadence" || sguid == "2d8cbd62-f04f-4b72-b1ec-b41c9ac9bbb7") {
							SaaSParams["patching_cadence"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "potentially exploited" || salias == "potentially_exploited" || sguid == "2c0e1cf5-bd7c-4859-ba54-674df3c76978") {
							SaaSParams["potentially_exploited"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "security incidents/breaches" || salias == "security_incidents_breaches" || sguid == "63beaccb-1ef6-40f3-a998-3a5d469095a4") {
							SaaSParams["data_breaches"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "server software" || salias == "server_software" || sguid == "eb0f9046-2a57-4f82-b0e7-936c242b5c63") {
							SaaSParams["server_software"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "spam propagation" || salias == "spam_propagation" || sguid == "ae1805d4-2405-4310-8d99-4fddbf624fd8") {
							SaaSParams["spam_propagation"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "spf domains" || salias == "spf_domains" || sguid == "e6eb290b-b986-4ada-b3aa-6f1bfcd6e14e") {
							SaaSParams["spf"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "tls/ssl certificates" || salias == "tls_ssl_certificates" || sguid == "e37379d7-c383-4efc-8be5-75c0b352073d") {
							SaaSParams["ssl_certificates"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "tls/ssl configurations" || salias == "tls_ssl_configurations" || sguid == "c61e3eec-7ab9-43d0-a1a5-e08b65c1f680") {
							SaaSParams["ssl_configurations"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "unsolicited communications" || salias == "unsolicited_communications" || sguid == "395a5113-fab9-4c71-bb12-7b1d60c33dde") {
							SaaSParams["unsolicited_comm"] = resultJSON[iField].RequestedObject.Id;
						} else if (sfieldname == "web application headers" || salias == "web_application_headers" || sguid == "99ac94c3-bcdc-4521-ae0c-260d773a1e91") {
							SaaSParams["application_security"] = resultJSON[iField].RequestedObject.Id; //Web Application Headers
						} else if (sfieldname == "bitsight tier" || salias == "bitsight_tier" || sguid == params["archer_bitsightportfolio_tier_guid"]) {
							SaaSParams["bitsight_tier"] = resultJSON[iField].RequestedObject.Id;

							ArcherValuesLists[ArcherValuesLists.length] = {
								"FieldName": "bitsight_tier",
								"ValuesListID": resultJSON[iField].RequestedObject.RelatedValuesListId,
								"Prefix": "bitsight_tier_",
							};
						}
					}

					if (bVerboseLogging === true) {
						var fs = require("fs");
						fs.writeFileSync("logs\\" + appPrefix + "\\" + "aParameters1.json", JSON.stringify(SaaSParams));
						LogInfo("Saved to logs\\" + appPrefix + "\\" + "aParameters1.json file");
					}

					LogInfo("SaaSParams: " + jsonToString(SaaSParams));

					getValuesListValues(0);
				} else {
					LogError("ERROR Obtaining Portfolio field definition: " + b);
				}
			}
		});
	};

	var getValuesListValues = function (currentValuesList) {
		LogInfo("----------------------------getValuesListValues----------------------------");
		LogInfo("Getting VL: " + ArcherValuesLists[currentValuesList].FieldName);
		var valueslistID = ArcherValuesLists[currentValuesList].ValuesListID;
		var url = params["archer_webroot"] + params["archer_rest_root"] + params["archer_valueslistvaluepath"] + valueslistID;

		var headers = {
			"Authorization": 'Archer session-id="' + sSessionToken + '"',
			"Content-type": "application/json",
			"Accept": "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
		};

		var postBody = {};

		LogInfo("URL8: " + url);

		webCall(url, "GET", headers, jsonToString(postBody), function (err, h, b) {
			if (err) {
				LogError("ERR8: " + err);
				//LogInfo("HEAD8: " + h);
				LogError("BODY8: " + b);
				callBack(true, null);
			} else {
				LogVerbose("ValuesList Full Definition: " + b);
				var resultJSON = JSON.parse(b);

				if (typeof resultJSON != "undefined" && typeof resultJSON[0].RequestedObject != "undefined") {
					LogVerbose("Archer Returned good values list data");

					var id = "";
					var name = "";
					var prefix = ArcherValuesLists[currentValuesList].Prefix;

					for (var i in resultJSON) {
						id = resultJSON[i].RequestedObject.Id;
						name = resultJSON[i].RequestedObject.Name;

						//Format name of value to remove spaces and /
						name = name.replace(/\//g, "");
						name = name.replace(/ /g, "");

						name = prefix + name; //Set the name of the parameter to fieldname_Value
						SaaSParams[name] = id; //Add to our parameters
					}

					//Iterate through if there were multiple
					//Did we hit the max?
					LogInfo("Current=" + currentValuesList + " Max=" + ArcherValuesLists.length);
					if (currentValuesList >= ArcherValuesLists.length - 1) {
						LogInfo("Hit maxResults of " + ArcherValuesLists.length);
						if (bVerboseLogging === true) {
							var fs = require("fs");
							fs.writeFileSync("logs\\" + appPrefix + "\\" + "aParameters2.json", JSON.stringify(SaaSParams));
							LogInfo("Saved to logs\\" + appPrefix + "\\" + "aParameters2.json file");
						}

						formatDataForSaaSUpload(callBack);
					} else {
						//Still have more values lists to iterate through...
						currentValuesList++; //Increment before running again
						LogInfo("Iterating through next ValuesList=" + currentValuesList);
						getValuesListValues(currentValuesList);
					}
				} else {
					LogError("ERROR Obtaining Portfolio field values list ids");
				}
			}
		});
	};

	//Start of all the functions to get ids
	getPortfolioModuleID(callBack);
};

/********************************************************/
/********	ReturnDataToArcher
/********************************************************/
var ReturnDataToArcher = function (callBack) {
	LogInfo("--------------Non-SaaS - ReturnDataToArcher-------------------------------");
	LogInfo("--------------Generating XML for Archer Records---------------------------");
	var start = Buffer.from("<Records>\n", "utf8");
	var xmlData = jsonArrayToXMLBuffer(aBitSightRatings, "Record");
	var end = Buffer.from("</Records>\n", "utf8");
	var returntoArcher = Buffer.concat([start, xmlData, end]);

	var sXMLReturn = returntoArcher.toString("utf8");
	//LogInfo("XML: " + sXMLReturn);

	LogInfo("Finish with callback...");
	callBack(false, sXMLReturn);
};

/********************************************************/
/********	formatDataForSaaSUpload
/********************************************************/
var formatDataForSaaSUpload = function (callBack) {
	LogInfo("----------------------formatDataForSaaSUpload START------------------------------");

	//var postBody = [];  //Array formatted as the postBody that will loop through to create records later

	var LevelID = SaaSParams["LevelID"];
	var GUID = SaaSParams["GUID"];
	var CompanyName = SaaSParams["CompanyName"];
	var CompanyID = SaaSParams["company_id"]; //5/20/2024 Added to bring in Company ID
	var last_total_risk_monitoring_update = SaaSParams["last_total_risk_monitoring_update"]; //Date used for the last time TRM date was updated in case they switch to a different subscription level.
	var domain = SaaSParams["domain"];
	var description = SaaSParams["description"];
	var industry = SaaSParams["industry"];
	var rating_date = SaaSParams["rating_date"];
	var rating_range = SaaSParams["rating_range"];
	var rating = SaaSParams["rating"];
	var subscripion_type = SaaSParams["subscripion_type"];
	var botnet_infections = SaaSParams["botnet_infections"];
	var spam_propagation = SaaSParams["spam_propagation"];
	var malware_servers = SaaSParams["malware_servers"];
	var unsolicited_comm = SaaSParams["unsolicited_comm"];
	var potentially_exploited = SaaSParams["potentially_exploited"];
	var spf = SaaSParams["spf"];
	var dkim = SaaSParams["dkim"];
	var ssl_certificates = SaaSParams["ssl_certificates"];
	var ssl_configurations = SaaSParams["ssl_configurations"];
	var open_ports = SaaSParams["open_ports"];
	var application_security = SaaSParams["application_security"];
	var patching_cadence = SaaSParams["patching_cadence"];
	var insecure_systems = SaaSParams["insecure_systems"];
	var server_software = SaaSParams["server_software"];
	var desktop_software = SaaSParams["desktop_software"];
	var mobile_software = SaaSParams["mobile_software"];
	var dnssec = SaaSParams["dnssec"];
	var mobile_application_security = SaaSParams["mobile_application_security"];
	var file_sharing = SaaSParams["file_sharing"];
	var data_breaches = SaaSParams["data_breaches"];
	var bitsight_tier = SaaSParams["bitsight_tier"];

	LogInfo("Count of aBitSightRatings: " + aBitSightRatings.length);

	if (aBitSightRatings.length == 0) {
		LogInfo("Nothing to send back to Archer...exiting.");
		LogSaaSSuccess();
	} else {
		for (var i in aBitSightRatings) {
			LogInfo("Company(" + i + ")=" + aBitSightRatings[i].CompanyName);

			var subscription_type_vl = GetSubscriptionTypeListValue(aBitSightRatings[i].subscription_type);
			var rating_range_vl = GetRangeTypeListValue(aBitSightRatings[i].range);
			var botnet_infections_vl = GetRatingListValue(aBitSightRatings[i].botnet_infections);
			var spam_propagation_vl = GetRatingListValue(aBitSightRatings[i].spam_propagation);
			var malware_servers_vl = GetRatingListValue(aBitSightRatings[i].malware_servers);
			var unsolicited_comm_vl = GetRatingListValue(aBitSightRatings[i].unsolicited_comm);
			var potentially_exploited_vl = GetRatingListValue(aBitSightRatings[i].potentially_exploited);
			var spf_vl = GetRatingListValue(aBitSightRatings[i].spf);
			var dkim_vl = GetRatingListValue(aBitSightRatings[i].dkim);
			var ssl_certificates_vl = GetRatingListValue(aBitSightRatings[i].ssl_certificates);
			var ssl_configurations_vl = GetRatingListValue(aBitSightRatings[i].ssl_configurations);
			var open_ports_vl = GetRatingListValue(aBitSightRatings[i].open_ports);
			var application_security_vl = GetRatingListValue(aBitSightRatings[i].application_security);
			var patching_cadence_vl = GetRatingListValue(aBitSightRatings[i].patching_cadence);
			var insecure_systems_vl = GetRatingListValue(aBitSightRatings[i].insecure_systems);
			var server_software_vl = GetRatingListValue(aBitSightRatings[i].server_software);
			var desktop_software_vl = GetRatingListValue(aBitSightRatings[i].desktop_software);
			var mobile_software_vl = GetRatingListValue(aBitSightRatings[i].mobile_software);
			var dnssec_vl = GetRatingListValue(aBitSightRatings[i].dnssec);
			var mobile_application_security_vl = GetRatingListValue(aBitSightRatings[i].mobile_application_security);
			var file_sharing_vl = GetRatingListValue(aBitSightRatings[i].file_sharing);
			var data_breaches_vl = GetRatingListValue(aBitSightRatings[i].data_breaches);
			var bitsight_tier_vl = GetBitSightTierListValue(aBitSightRatings[i].BitSightTier);
			var contentID = aBitSightRatings[i].ContentID;

			if (aBitSightRatings[i].subscription_type == "Not Subscribed") {
				LogInfo("-Not Subscribed...");
				aFinalPostBody[i] = {
					"Content": {
						"LevelId": LevelID,
						"Tag": "BitSightPortfolio",
						"FieldContents": {
							[GUID]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "Company GUID",
								"Value": aBitSightRatings[i].GUID,
								"FieldID": GUID,
							},
							[CompanyName]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "CompanyName",
								"Value": aBitSightRatings[i].CompanyName,
								"FieldID": CompanyName,
							},
							[CompanyID]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "CompanyID",
								"Value": aBitSightRatings[i].custom_id,
								"FieldID": CompanyID,
							},
							[domain]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "domain",
								"Value": aBitSightRatings[i].primary_domain,
								"FieldID": domain,
							},
							[description]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "description",
								"Value": aBitSightRatings[i].description,
								"FieldID": description,
							},
							[industry]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "industry",
								"Value": aBitSightRatings[i].industry,
								"FieldID": industry,
							},
							[subscripion_type]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "subscripion_type",
								"Value": {
									"ValuesListIds": [subscription_type_vl],
								},
								"FieldID": subscripion_type,
							},
							[bitsight_tier]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "bitsight_tier",
								"Value": {
									"ValuesListIds": [bitsight_tier_vl],
								},
								"FieldID": bitsight_tier,
							},
						},
					},
				}; //End of postBody
			} else if (aBitSightRatings[i].subscription_type == "Total Risk Monitoring") {
				LogInfo("-TRM...");
				aFinalPostBody[i] = {
					"Content": {
						"LevelId": LevelID,
						"Tag": "BitSightPortfolio",
						"FieldContents": {
							[GUID]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "Company GUID",
								"Value": aBitSightRatings[i].GUID,
								"FieldID": GUID,
							},
							[CompanyName]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "CompanyName",
								"Value": aBitSightRatings[i].CompanyName,
								"FieldID": CompanyName,
							},
							[CompanyID]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "CompanyID",
								"Value": aBitSightRatings[i].custom_id,
								"FieldID": CompanyID,
							},
							[domain]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "domain",
								"Value": aBitSightRatings[i].primary_domain,
								"FieldID": domain,
							},
							[description]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "description",
								"Value": aBitSightRatings[i].description,
								"FieldID": description,
							},
							[industry]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "industry",
								"Value": aBitSightRatings[i].industry,
								"FieldID": industry,
							},
							[rating_date]: {
								"Type": ArcherInputTypes["date"],
								"Tag": "rating_date",
								"Value": aBitSightRatings[i].rating_date,
								"FieldID": rating_date,
							},
							[last_total_risk_monitoring_update]: {
								"Type": ArcherInputTypes["date"],
								"Tag": "last_total_risk_monitoring_update",
								"Value": aBitSightRatings[i].rating_date,
								"FieldID": last_total_risk_monitoring_update,
							},
							[rating]: {
								"Type": ArcherInputTypes["numeric"],
								"Tag": "rating",
								"Value": aBitSightRatings[i].rating,
								"FieldID": rating,
							},
							[subscripion_type]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "subscripion_type",
								"Value": {
									"ValuesListIds": [subscription_type_vl],
								},
								"FieldID": subscripion_type,
							},
							[rating_range]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "rating_range",
								"Value": {
									"ValuesListIds": [rating_range_vl],
								},
								"FieldID": rating_range,
							},
							[botnet_infections]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "botnet_infections",
								"Value": {
									"ValuesListIds": [botnet_infections_vl],
								},
								"FieldID": botnet_infections,
							},
							[spam_propagation]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "spam_propagation",
								"Value": {
									"ValuesListIds": [spam_propagation_vl],
								},
								"FieldID": spam_propagation,
							},
							[malware_servers]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "malware_servers",
								"Value": {
									"ValuesListIds": [malware_servers_vl],
								},
								"FieldID": malware_servers,
							},
							[unsolicited_comm]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "unsolicited_comm",
								"Value": {
									"ValuesListIds": [unsolicited_comm_vl],
								},
								"FieldID": unsolicited_comm,
							},
							[potentially_exploited]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "potentially_exploited",
								"Value": {
									"ValuesListIds": [potentially_exploited_vl],
								},
								"FieldID": potentially_exploited,
							},
							[spf]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "spf",
								"Value": {
									"ValuesListIds": [spf_vl],
								},
								"FieldID": spf,
							},
							[dkim]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "dkim",
								"Value": {
									"ValuesListIds": [dkim_vl],
								},
								"FieldID": dkim,
							},
							[ssl_certificates]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "ssl_certificates",
								"Value": {
									"ValuesListIds": [ssl_certificates_vl],
								},
								"FieldID": ssl_certificates,
							},
							[ssl_configurations]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "ssl_configurations",
								"Value": {
									"ValuesListIds": [ssl_configurations_vl],
								},
								"FieldID": ssl_configurations,
							},
							[open_ports]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "open_ports",
								"Value": {
									"ValuesListIds": [open_ports_vl],
								},
								"FieldID": open_ports,
							},
							[application_security]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "application_security",
								"Value": {
									"ValuesListIds": [application_security_vl],
								},
								"FieldID": application_security,
							},
							[patching_cadence]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "patching_cadence",
								"Value": {
									"ValuesListIds": [patching_cadence_vl],
								},
								"FieldID": patching_cadence,
							},
							[insecure_systems]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "insecure_systems",
								"Value": {
									"ValuesListIds": [insecure_systems_vl],
								},
								"FieldID": insecure_systems,
							},
							[server_software]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "server_software",
								"Value": {
									"ValuesListIds": [server_software_vl],
								},
								"FieldID": server_software,
							},
							[desktop_software]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "desktop_software",
								"Value": {
									"ValuesListIds": [desktop_software_vl],
								},
								"FieldID": desktop_software,
							},
							[mobile_software]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "mobile_software",
								"Value": {
									"ValuesListIds": [mobile_software_vl],
								},
								"FieldID": mobile_software,
							},
							[dnssec]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "dnssec",
								"Value": {
									"ValuesListIds": [dnssec_vl],
								},
								"FieldID": dnssec,
							},
							[mobile_application_security]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "mobile_application_security",
								"Value": {
									"ValuesListIds": [mobile_application_security_vl],
								},
								"FieldID": mobile_application_security,
							},
							[file_sharing]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "file_sharing",
								"Value": {
									"ValuesListIds": [file_sharing_vl],
								},
								"FieldID": file_sharing,
							},
							[data_breaches]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "data_breaches",
								"Value": {
									"ValuesListIds": [data_breaches_vl],
								},
								"FieldID": data_breaches,
							},
							[bitsight_tier]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "bitsight_tier",
								"Value": {
									"ValuesListIds": [bitsight_tier_vl],
								},
								"FieldID": bitsight_tier,
							},
						},
					},
				}; //End of postBody
			} else {
				LogInfo("-RM...");
				aFinalPostBody[i] = {
					"Content": {
						"LevelId": LevelID,
						"Tag": "BitSightPortfolio",
						"FieldContents": {
							[GUID]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "Company GUID",
								"Value": aBitSightRatings[i].GUID,
								"FieldID": GUID,
							},
							[CompanyName]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "CompanyName",
								"Value": aBitSightRatings[i].CompanyName,
								"FieldID": CompanyName,
							},
							[CompanyID]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "CompanyID",
								"Value": aBitSightRatings[i].custom_id,
								"FieldID": CompanyID,
							},
							[domain]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "domain",
								"Value": aBitSightRatings[i].primary_domain,
								"FieldID": domain,
							},
							[description]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "description",
								"Value": aBitSightRatings[i].description,
								"FieldID": description,
							},
							[industry]: {
								"Type": ArcherInputTypes["text"],
								"Tag": "industry",
								"Value": aBitSightRatings[i].industry,
								"FieldID": industry,
							},
							[rating_date]: {
								"Type": ArcherInputTypes["date"],
								"Tag": "rating_date",
								"Value": aBitSightRatings[i].rating_date,
								"FieldID": rating_date,
							},
							[rating]: {
								"Type": ArcherInputTypes["numeric"],
								"Tag": "rating",
								"Value": aBitSightRatings[i].rating,
								"FieldID": rating,
							},
							[subscripion_type]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "subscripion_type",
								"Value": {
									"ValuesListIds": [subscription_type_vl],
								},
								"FieldID": subscripion_type,
							},
							[rating_range]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "rating_range",
								"Value": {
									"ValuesListIds": [rating_range_vl],
								},
								"FieldID": rating_range,
							},
							[bitsight_tier]: {
								"Type": ArcherInputTypes["valueslist"],
								"Tag": "bitsight_tier",
								"Value": {
									"ValuesListIds": [bitsight_tier_vl],
								},
								"FieldID": bitsight_tier,
							},
						},
					},
				}; //End of postBody
			}
			//If the Content ID exists, then we need to add it to the body. The call to Archer will use this to determine POST vs. PUT
			if (contentID != null && contentID != "") {
				aFinalPostBody[i].Content.Id = contentID;
			}

			//LogVerbose("aFinalPostBody[" + i + "] = " +  jsonToString(aFinalPostBody[i]));
		} //End of for loop for iteration

		return SaaSuploadToArcher(callBack);
	} //End of if statement when there is something to return to Archer
}; //End of formatDataForSaaSUpload function

function GetRatingListValue(sLetter) {
	if (typeof sLetter == "undefined" || sLetter == null || sLetter == "" || sLetter == "N/A") {
		return SaaSParams["rating_NA"];
	}
	sLetter = sLetter.toLowerCase();
	if (sLetter == "a") {
		return SaaSParams["rating_A"];
	} else if (sLetter == "b") {
		return SaaSParams["rating_B"];
	} else if (sLetter == "c") {
		return SaaSParams["rating_C"];
	} else if (sLetter == "d") {
		return SaaSParams["rating_D"];
	} else if (sLetter == "f") {
		return SaaSParams["rating_F"];
	} else {
		return SaaSParams["rating_NA"];
	}
}

function GetBitSightTierListValue(tier) {
	if (typeof tier == "undefined" || tier == null || tier == "") {
		return null;
	}

	//Format name of value to remove spaces and /
	//We did this when we added the tiers to the SaasParams object so this will help with matching
	tier = tier.replace(/\//g, "");
	tier = tier.replace(/ /g, "");

	let kTierName = "bitsight_tier_" + tier;

	if (typeof SaaSParams[kTierName] != "undefined" && SaaSParams[kTierName] != null) {
		LogInfo("GetBitSightTierListValue() tier=" + tier + " kTierName=" + kTierName + " ArcherValuesListID=" + SaaSParams[kTierName]);
		return SaaSParams[kTierName];
	} else {
		LogInfo("GetBitSightTierListValue() tier=" + tier + " NOT FOUND!");
		return null;
	}
}

function GetSubscriptionTypeListValue(sSubscriptionType) {
	if (typeof sSubscriptionType == "undefined" || sSubscriptionType == null || sSubscriptionType == "" || sSubscriptionType == "N/A") {
		LogInfo("Subscription Type undefined, null, empty, or N/A");
		return SaaSParams["subscriptionType_NotSubscribed"]; //Need to set to Not Subscribed or could impact headline ratings process
	}
	sSubscriptionType = sSubscriptionType.toLowerCase();
	if (sSubscriptionType == "total risk monitoring") {
		return SaaSParams["subscriptionType_TotalRiskMonitoring"];
	} else if (sSubscriptionType == "risk monitoring") {
		return SaaSParams["subscriptionType_RiskMonitoring"];
	} else if (sSubscriptionType == "not subscribed") {
		//This is when they get unsubscribed
		return SaaSParams["subscriptionType_NotSubscribed"];
	} else {
		LogWarn("Subscription Type Unknown: " + sSubscriptionType);
		return SaaSParams["subscriptionType_NotSubscribed"]; //Need to set to Not Subscribed or could impact headline ratings process
	}
}

function GetRangeTypeListValue(sRange) {
	if (typeof sRange == "undefined" || sRange == null || sRange == "" || sRange == "N/A") {
		return "";
	}
	sRange = sRange.toLowerCase();
	if (sRange == "basic") {
		return SaaSParams["range_Basic"];
	} else if (sRange == "intermediate") {
		return SaaSParams["range_Intermediate"];
	} else if (sRange == "advanced") {
		return SaaSParams["range_Advanced"];
	} else {
		LogWarn("Range Type Unknown: " + sRange);
		return "";
	}
}

/********************************************************/
/********	SaaSuploadToArcher
/********************************************************/
var SaaSuploadToArcher = function (callBack) {
	LogInfo("STARTING ON RECORD #: " + iCurrentCompany);

	var sMETHOD = "";
	if (aFinalPostBody[iCurrentCompany].Content.Id == null || aFinalPostBody[iCurrentCompany].Content.Id == "") {
		sMETHOD = "POST";
		LogInfo("----------------------------CREATE RECORD----------------------------");
	} else {
		sMETHOD = "PUT";
		LogInfo("----------------------------UPDATE RECORD----------------------------");
		LogInfo("ContentID: " + aFinalPostBody[iCurrentCompany].Content.Id + " Company: " + aBitSightRatings[iCurrentCompany].CompanyName);
	}

	var url = params["archer_webroot"] + params["archer_rest_root"] + params["archer_contentpath"];

	var headers = {
		"Authorization": 'Archer session-id="' + sSessionToken + '"',
		"Content-type": "application/json",
		"Accept": "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
	};

	LogInfo("URL9: " + url);
	//LogVerbose('BODY9: ' + jsonToString(aFinalPostBody[iCurrentCompany])); //Since this is being saved as a file, no need to output to console.

	if (bVerboseLogging === true) {
		var fs = require("fs");
		fs.writeFileSync("logs\\" + appPrefix + "\\" + "JSON_PostBody_" + iCurrentCompany + ".json", jsonToString(aFinalPostBody[iCurrentCompany]));
		LogInfo("Saved to logs\\" + appPrefix + "\\" + "JSON_PostBody_" + iCurrentCompany + ".json file");
	}

	webCall(url, sMETHOD, headers, jsonToString(aFinalPostBody[iCurrentCompany]), function (err, h, b) {
		if (err) {
			//If skipping when an error is true, then skip over it and log the warning. Otherwise, terminate the entire process.
			if (bSkipCompanyIfError) {
				LogWarn("bSkipCompanyIfError is set to true.");
				LogWarn("ERR9: " + err);
				LogWarn("BODY9: " + b);

				//Iterate through if there were multiple records to create
				//Did we hit the max?
				LogInfo("Current=" + iCurrentCompany + " Max=" + aFinalPostBody.length);
				if (iCurrentCompany >= aFinalPostBody.length - 1) {
					LogInfo("Hit maxResults of " + aFinalPostBody.length);
					LogSaaSSuccess();
					LogInfo("SUCCESS: SaaS Records Created");
					return true;
				} else {
					//Still have more records to iterate through...
					LogInfo("Iterating through next record=" + iCurrentCompany);
					iCurrentCompany++; //Increment before running again
					return SaaSuploadToArcher(callBack);
				}
			} else {
				LogError("ERR9: " + err);
				//LogError(" HEAD9: " + h);
				LogError("BODY9: " + b);
				callBack(true, null);
			}
		} else {
			//LogInfo("Raw Report Results: " + b);
			var resultJSON = JSON.parse(b);

			//Just for testing...save to file...
			if (bVerboseLogging === true) {
				var fs = require("fs");
				fs.writeFileSync("logs\\" + appPrefix + "\\" + "resultJSON" + iCurrentCompany + ".json", JSON.stringify(resultJSON));
				LogInfo("Saved to logs\\" + appPrefix + "\\" + "resultJSON" + iCurrentCompany + ".json file");
			}

			var iContentID = resultJSON.RequestedObject.Id; //Get the content ID
			LogInfo("iContentID created/updated: " + iContentID);

			//Iterate through if there were multiple records to create
			//Did we hit the max?
			LogInfo("Current=" + iCurrentCompany + " Max=" + aFinalPostBody.length);
			if (iCurrentCompany >= aFinalPostBody.length - 1) {
				LogInfo("Hit maxResults of " + aFinalPostBody.length);
				LogSaaSSuccess();
				LogInfo("SUCCESS: SaaS Records Created");
				return true;
			} else {
				//Still have more records to iterate through...
				LogInfo("Iterating through next record=" + iCurrentCompany);
				iCurrentCompany++; //Increment before running again
				return SaaSuploadToArcher(callBack);
			}
		}
	});
}; //End of uploadToArcher function

/************************************************************************************************************************************************************************************************************************************/
/******** DATAFEED BEGINS HERE
/************************************************************************************************************************************************************************************************************************************/
var lrt = init();
ArcherAPI_Login(dfCallback);
